(function (global) {
  "use strict";

  var d = document;

  // ---------- DOM ----------
  function $(id) { return d.getElementById(id); }

  // ---------- toast / modal helpers ----------
  function getToastEl() { return $("toast"); }
  function getModalEls() {
    var backdrop = $("modalBackdrop");
    return {
      backdrop: backdrop,
      modal: backdrop ? backdrop.querySelector(".modal") : null,
      title: $("modalTitle"),
      body: $("modalBody")
    };
  }
  function showToast(msg) {
    var toast = getToastEl();
    if (!toast) return;
    toast.textContent = msg;
    toast.classList.add("show");
    setTimeout(function () { toast.classList.remove("show"); }, 2600);
  }
  function showModal(title, message, variant) {
    var m = getModalEls();
    if (!m.backdrop) return;
    if (m.modal) {
      m.modal.classList.remove("success");
      if (variant === "success") m.modal.classList.add("success");
    }
    if (m.title) m.title.textContent = title;
    if (m.body) m.body.textContent = message;
    m.backdrop.style.display = "flex";
  }
  function hideModal() {
    var m = getModalEls();
    if (m.backdrop) m.backdrop.style.display = "none";
  }
  d.addEventListener("click", function (e) {
    if (e.target && e.target.id === "modalCloseBtn") { hideModal(); return; }
    var m = getModalEls();
    if (m.backdrop && e.target === m.backdrop) { hideModal(); }
  });
  d.addEventListener("keydown", function (e) {
    var m = getModalEls();
    if (e.key === "Escape" && m.backdrop && m.backdrop.style.display === "flex") hideModal();
  });

  // ---------- busy / overlay ----------
  function getOverlay() { return $("loadingOverlay"); }
  function restartSpinner() {
    var overlay = getOverlay();
    if (!overlay) return;
    var spinner = overlay.querySelector(".spinner");
    if (!spinner) return;
    var fresh = spinner.cloneNode(true);
    spinner.replaceWith(fresh);
  }
  async function runWithBusy(fn, controls) {
    var overlay = getOverlay();
    var list = (controls || []).map(function (x) { return typeof x === "string" ? $(x) : x; }).filter(Boolean);
    list.forEach(function (el) { el.disabled = true; });
    if (overlay) {
      overlay.style.display = "flex";
      restartSpinner();
      await new Promise(function (r) { requestAnimationFrame(r); });
      await new Promise(function (r) { requestAnimationFrame(r); });
    }
    try { await fn(); }
    finally {
      list.forEach(function (el) { el.disabled = false; });
      if (overlay) overlay.style.display = "none";
    }
  }

  // ---------- clipboard ----------
  function fallbackCopy(text, onSuccess, onError) {
    try {
      var ta = d.createElement("textarea");
      ta.value = text;
      ta.style.position = "fixed";
      ta.style.top = "-9999px";
      d.body.appendChild(ta);
      ta.focus(); ta.select();
      var ok = d.execCommand("copy");
      d.body.removeChild(ta);
      ok ? (onSuccess && onSuccess()) : (onError && onError());
    } catch (err) {
      if (onError) onError(err);
    }
  }
  function copyToClipboard(text, successMsg) {
    if (!text) { showToast("Nothing to copy"); return; }
    if (navigator.clipboard && window.isSecureContext) {
      navigator.clipboard.writeText(text).then(function () {
        showToast(successMsg || "Copied");
      }).catch(function () {
        fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                          function () { showToast("Copy failed"); });
      });
    } else {
      fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                        function () { showToast("Copy failed"); });
    }
  }

  // ---------- randomness ----------
  var MODES = {
    classic: {
      alphabet: "ADEFHKMNPRTWX347",
      entropyBits: 128
    },
    extreme: {
      alphabet: "ABCDEFGHJKLMNPQRSTUVWXYZabcdefghkmnpqrstuvwxyz2346789",
      entropyBits: 185
    }
  };

  function hasWebCrypto() {
    return typeof window !== "undefined" && window.crypto && typeof window.crypto.getRandomValues === "function";
  }

  function evaluateRandomResult(info) {
    var hasSecure = !!info.hasSecure;
    var mathRandomUsed = !!info.mathRandomUsed;
    if (!hasSecure) {
      return {
        ok: false, variant: null,
        message: "❌ ERROR: INSECURE RANDOMNESS ❌ This browser did NOT provide crypto.getRandomValues(). Use a modern browser, offline."
      };
    }
    if (mathRandomUsed) {
      return {
        ok: false, variant: null,
        message: "❌ ERROR: INSECURE RANDOMNESS ❌ The generator fell back to Math.random(). Use a different browser/build."
      };
    }
    return { ok: true, variant: "success", message: "✅ Secure randomness OK." };
  }

  function generatePassword(alphabet) {
    var length = 32;
    var secure = hasWebCrypto();
    var buf = new Uint8Array(length);
    var mathRandomUsed = false;

    if (secure) {
      window.crypto.getRandomValues(buf);
    } else {
      mathRandomUsed = true;
      for (var i = 0; i < length; i++) buf[i] = Math.floor(Math.random() * 256);
    }

    var out = "";
    var mod = alphabet.length;
    for (var j = 0; j < length; j++) out += alphabet[buf[j] % mod];
    buf.fill(0);

    var groups = [];
    for (var k = 0; k < out.length; k += 4) groups.push(out.slice(k, k + 4));
    var formatted = groups.join("-");

    var assessment = evaluateRandomResult({ hasSecure: secure, mathRandomUsed: mathRandomUsed });
    return { value: formatted, ok: assessment.ok, variant: assessment.variant, message: assessment.message };
  }

  // ---------- PASSGEN APP ----------
  var pgOutput = $("pgOutput");
  var copyBtn = $("copyBtn");
  var generateBtn = $("generateBtn");
  var clearBtn = $("clearBtn");
  var statusText = $("statusText");
  var modeClassic = $("modeClassic");
  var modeExtreme = $("modeExtreme");
  var alphabetInfo = $("alphabetInfo");
  var pageDesc = $("pageDesc");

  function currentModeKey() {
    return modeExtreme.checked ? "extreme" : "classic";
  }

  function refreshInfo() {
    var m = MODES[currentModeKey()];
    if (alphabetInfo) {
      alphabetInfo.innerHTML = 'Alphabet: <code>' + m.alphabet + '</code>. Length: 32, Entropy: ' + m.entropyBits + '-bit';
    }
    if (pageDesc) {
      pageDesc.textContent = "Generate a 32‑character (" + m.entropyBits + "-bit entropy) password fully offline.";
    }
  }

  function clearFields() {
    pgOutput.value = "";
    copyBtn.disabled = true;
    statusText.textContent = "Cleared.";
    statusText.classList.remove("bad","ok");
  }

  function updateStatus(text, isOk) {
    statusText.textContent = text;
    statusText.classList.remove("bad", "ok");
    statusText.classList.add(isOk ? "ok" : "bad");
  }

  generateBtn.addEventListener("click", function () {
    runWithBusy(async function () {
      var m = MODES[currentModeKey()];
      var rnd = generatePassword(m.alphabet);
      pgOutput.value = rnd.value;
      copyBtn.disabled = !rnd.value;
      updateStatus(rnd.message, rnd.ok);
      if (rnd.ok) {
        showToast("Secure random password generated (" + m.entropyBits + "-bit)");
      } else {
        showModal("Randomness ERROR", rnd.message);
      }
    }, [generateBtn, clearBtn]);
  });

  copyBtn.addEventListener("click", function () {
    copyToClipboard(pgOutput.value, "Password copied");
  });

  clearBtn.addEventListener("click", clearFields);
  modeClassic.addEventListener("change", refreshInfo);
  modeExtreme.addEventListener("change", refreshInfo);

  // init
  refreshInfo();

})(typeof window !== "undefined" ? window : this);
