"use strict";

(function (global) {
    "use strict";

    var d = document;

    function $(id) { return d.getElementById(id); }

    function getToastEl() { return $("toast"); }

    function getModalEls() {
        var backdrop = $("modalBackdrop");
        return {
            backdrop: backdrop,
            modal: backdrop ? backdrop.querySelector(".modal") : null,
            title: $("modalTitle"),
            body: $("modalBody")
        };
    }

    function showToast(msg) {
        var toast = getToastEl();
        if (!toast) return;
        toast.textContent = msg;
        toast.classList.add("show");
        setTimeout(function () { toast.classList.remove("show"); }, 2600);
    }

    function showModal(title, message, variant) {
        var m = getModalEls();
        if (!m.backdrop) return;
        if (m.modal) {
            m.modal.classList.remove("success");
            if (variant === "success") m.modal.classList.add("success");
        }
        if (m.title) m.title.textContent = title;
        if (m.body) m.body.textContent = message;
        m.backdrop.style.display = "flex";
    }

    function hideModal() {
        var m = getModalEls();
        if (m.backdrop) m.backdrop.style.display = "none";
    }

    function getOverlay(elOrId) {
        if (elOrId && elOrId instanceof HTMLElement) return elOrId;
        if (typeof elOrId === "string") return $(elOrId);
        return $("loadingOverlay");
    }

    function toElementArray(list) {
        if (!list) return [];
        return list.map(function (item) {
            if (!item) return null;
            if (item instanceof HTMLElement) return item;
            if (typeof item === "string") return $(item);
            return null;
        }).filter(Boolean);
    }

    function restartSpinner(overlayEl) {
        var overlay = getOverlay(overlayEl);
        if (!overlay) return;
        var spinner = overlay.querySelector(".spinner");
        if (!spinner) return;
        var fresh = spinner.cloneNode(true);
        spinner.replaceWith(fresh);
    }

    function setControlsDisabled(controls, disabled) {
        controls.forEach(function (el) { if (el) el.disabled = disabled; });
    }

    async function runWithBusy(fn, options) {
        options = options || {};
        var overlay = getOverlay(options.overlay);
        var controls = toElementArray(options.controls || []);
        setControlsDisabled(controls, true);
        if (overlay) {
            overlay.style.display = "flex";
            restartSpinner(overlay);
            await new Promise(function (r) { requestAnimationFrame(r); });
            await new Promise(function (r) { requestAnimationFrame(r); });
        }
        try { await fn(); }
        finally {
            setControlsDisabled(controls, false);
            if (overlay) overlay.style.display = "none";
        }
    }

    function fallbackCopy(text, onSuccess, onError) {
        try {
            var ta = d.createElement("textarea");
            ta.value = text;
            ta.style.position = "fixed";
            ta.style.top = "-9999px";
            d.body.appendChild(ta);
            ta.focus();
            ta.select();
            var ok = d.execCommand("copy");
            d.body.removeChild(ta);
            ok ? (onSuccess && onSuccess()) : (onError && onError());
        } catch (err) { onError && onError(err); }
    }

    function copyToClipboard(text, successMsg) {
        if (!text) { showToast("Nothing to copy"); return; }
        if (navigator.clipboard && window.isSecureContext) {
            navigator.clipboard.writeText(text)
                .then(function () { showToast(successMsg || "Copied"); })
                .catch(function () {
                    fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                                      function () { showToast("Copy failed"); });
                });
        } else {
            fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                              function () { showToast("Copy failed"); });
        }
    }

    function secureWipe() {
        for (var i = 0; i < arguments.length; i++) {
            var it = arguments[i];
            if (!it) continue;
            if (it instanceof Uint8Array) it.fill(0);
            else if (it instanceof ArrayBuffer) new Uint8Array(it).fill(0);
        }
    }

    function hasWebCrypto() {
        return typeof window !== "undefined" && window.crypto && typeof window.crypto.getRandomValues === "function";
    }

    function evaluateRandomResult(label, info) {
        var hasSecure = !!info.hasSecure;
        var mathRandomUsed = !!info.mathRandomUsed;
        if (!hasSecure) {
            return {
                ok: false,
                variant: null,
                message:
                    "❌ THIS " + label + " IS NOT SECURE ❌ This browser did NOT provide crypto.getRandomValues(). Use a modern browser, offline."
            };
        }
        if (mathRandomUsed) {
            return {
                ok: false,
                variant: null,
                message:
                    "❌ THIS " + label + " IS NOT SECURE ❌ The library fell back to Math.random(). Use another browser/build."
            };
        }
        return { ok: true, variant: "success", message: "✅ Secure " + label.toLowerCase() + " generated successfully." };
    }

    var DEFAULT_PASSWORD_CHARSET = "ADEFHKMNPRTWX347";

    function generateRandomString(opts) {
        opts = opts || {};
        var length = opts.length || 32;
        var charset = opts.charset || DEFAULT_PASSWORD_CHARSET;
        var label   = opts.label   || "VALUE";

        var secure = hasWebCrypto();
        var buf = new Uint8Array(length);
        var mathRandomUsed = false;

        if (secure) {
            window.crypto.getRandomValues(buf);
        } else {
            mathRandomUsed = true;
            for (var i = 0; i < length; i++) buf[i] = Math.floor(Math.random() * 256);
        }

        var raw = "";
        var mod = charset.length;
        for (var j = 0; j < length; j++) raw += charset[buf[j] % mod];

        secureWipe(buf);

        var groups = [];
        for (var k = 0; k < raw.length; k += 4) groups.push(raw.slice(k, k + 4));
        var formatted = groups.join("-");

        var assessment = evaluateRandomResult(label, { hasSecure: secure, mathRandomUsed: mathRandomUsed });

        return { value: formatted, ok: assessment.ok, variant: assessment.variant, message: assessment.message };
    }

    function bytesToHex(bytes) {
        return Array.prototype.map.call(bytes, function (b) { return b.toString(16).padStart(2, "0"); }).join("");
    }

    function hexToBytes(hex) {
        if (!hex) return new Uint8Array(0);
        var clean = String(hex).replace(/[^0-9a-f]/gi, "").toLowerCase();
        if (clean.length % 2 !== 0) throw new Error("Hex length must be even.");
        var out = new Uint8Array(clean.length / 2);
        for (var i = 0; i < clean.length; i += 2) out[i / 2] = parseInt(clean.substr(i, 2), 16);
        return out;
    }

    function toHex(data) {
        if (data instanceof ArrayBuffer) return bytesToHex(new Uint8Array(data));
        if (data instanceof Uint8Array) return bytesToHex(data);
        return "";
    }

    function fromHex(hex) { return hexToBytes(hex).buffer; }

    var _encoder = new TextEncoder();
    var _decoder = new TextDecoder("utf-8");
    function encodeText(str) { return _encoder.encode(str); }
    function decodeText(buf) { return _decoder.decode(buf); }

    var AppShared = {
        showToast: showToast,
        showModal: showModal,
        hideModal: hideModal,
        restartSpinner: restartSpinner,
        runWithBusy: runWithBusy,
        copyToClipboard: copyToClipboard,
        fallbackCopy: fallbackCopy,
        secureWipe: secureWipe,
        hasWebCrypto: hasWebCrypto,
        evaluateRandomResult: evaluateRandomResult,
        generateRandomString: generateRandomString,
        DEFAULT_PASSWORD_CHARSET: DEFAULT_PASSWORD_CHARSET,
        bytesToHex: bytesToHex,
        hexToBytes: hexToBytes,
        toHex: toHex,
        fromHex: fromHex,
        encodeText: encodeText,
        decodeText: decodeText
    };
    global.AppShared = AppShared;

    d.addEventListener("click", function (e) {
        var m = getModalEls();
        if (e.target && e.target.id === "modalCloseBtn") { hideModal(); return; }
        if (m.backdrop && e.target === m.backdrop) hideModal();
    });

    d.addEventListener("keydown", function (e) {
        var m = getModalEls();
        if (e.key === "Escape" && m.backdrop && m.backdrop.style.display === "flex") hideModal();
    });
})(typeof window !== "undefined" ? window : this);

(function (global) {
    "use strict";
    var AppShared = global.AppShared;

    function arrayToNewlineText(arr) {
        if (!Array.isArray(arr)) return "";
        return arr.map(function (s) { return String(s == null ? "" : s); }).join("\n");
    }

    function isArrayOfStrings(val) {
        if (!Array.isArray(val)) return false;
        for (var i = 0; i < val.length; i++) if (typeof val[i] !== "string") return false;
        return true;
    }

    AppShared.arrayToNewlineText = arrayToNewlineText;
    AppShared.isArrayOfStrings = isArrayOfStrings;
})(typeof window !== "undefined" ? window : this);

(() => {
    'use strict';

    const AS = self.AppShared;
    const wipe = AS.secureWipe;

    const canonicalJson = (obj) => {
        if (obj === null || typeof obj !== 'object') return obj;
        if (Array.isArray(obj)) return obj.map(canonicalJson);
        const out = {};
        for (const k of Object.keys(obj).sort()) out[k] = canonicalJson(obj[k]);
        return out;
    };
    const canonicalStringify = (obj) => JSON.stringify(canonicalJson(obj));

    const sha256 = async (inputBytes) => {
        const d = await crypto.subtle.digest('SHA-256', inputBytes);
        return new Uint8Array(d);
    };

    const hmacSha256Sign = async (keyBytes, msgBytes) => {
        const key = await crypto.subtle.importKey('raw', keyBytes, { name: 'HMAC', hash: 'SHA-256' }, false, ['sign']);
        const sig = await crypto.subtle.sign('HMAC', key, msgBytes);
        return new Uint8Array(sig);
    };

    const ctEqual = (a, b) => {
        const la = a.length >>> 0;
        const lb = b.length >>> 0;
        let diff = (la ^ lb) >>> 0;
        const L = la > lb ? la : lb;
        for (let i = 0; i < L; i++) {
            const ai = i < la ? a[i] : 0;
            const bi = i < lb ? b[i] : 0;
            diff |= (ai ^ bi);
        }
        return diff === 0;
    };

    const deriveMacKeyFromCek = async (cekBytes) => {
        const label = AS.encodeText('shamircipher-meta-mac-key');
        const buf = new Uint8Array(cekBytes.length + label.length);
        buf.set(cekBytes, 0);
        buf.set(label, cekBytes.length);
        const macKey = await sha256(buf);
        wipe(buf);
        return macKey;
    };

    const computePayloadMac = async (cekBytes, payloadNoMac) => {
        const macKey = await deriveMacKeyFromCek(cekBytes);
        try {
            const canon = canonicalStringify(payloadNoMac);
            const bytes = AS.encodeText(canon);
            const sig = await hmacSha256Sign(macKey, bytes);
            wipe(bytes);
            return sig;
        } finally {
            wipe(macKey);
        }
    };

    const verifyPayloadMac = async (cekBytes, payloadNoMac, expectedMacHex) => {
        const got = await computePayloadMac(cekBytes, payloadNoMac);
        const exp = new Uint8Array(AS.fromHex(expectedMacHex));
        try {
            return ctEqual(got, exp);
        } finally {
            wipe(got);
            wipe(exp);
        }
    };

    const importAesKey = (raw) => crypto.subtle.importKey('raw', raw, 'AES-GCM', false, ['encrypt', 'decrypt']);

    const aesGcmEncrypt = async (keyBytes, ivBytes, ptBytes, aad) => {
        const key = await importAesKey(keyBytes);
        const algo = { name: 'AES-GCM', iv: ivBytes, tagLength: 128, ...(aad ? { additionalData: aad } : {}) };
        const ct = await crypto.subtle.encrypt(algo, key, ptBytes);
        return new Uint8Array(ct);
    };

    const aesGcmDecrypt = async (keyBytes, ivBytes, ctBytes, aad) => {
        const key = await importAesKey(keyBytes);
        const algo = { name: 'AES-GCM', iv: ivBytes, tagLength: 128, ...(aad ? { additionalData: aad } : {}) };
        const pt = await crypto.subtle.decrypt(algo, key, ctBytes);
        return new Uint8Array(pt);
    };

    const randBytes = (len) => {
        if (!AS.hasWebCrypto()) throw new Error('CSPRNG unavailable (crypto.getRandomValues)');
        const b = new Uint8Array(len);
        crypto.getRandomValues(b);
        return b;
    };

    const MAX_ARGON2_MEM_KB = 1_048_576;
    const MAX_ARGON2_ITERS  = 64;
    const MAX_ARGON2_PAR    = 8;

    const MIN_ARGON2_MEM_KB = 131_072; // 128 MB
    const MIN_ARGON2_ITERS  = 4;

    let _lastKdfReport = null;
    const getLastKdfReport = () => (_lastKdfReport ? { ..._lastKdfReport } : null);

    const deriveKeyArgon2id = async (password, saltBytes, params) => {
        if (!self.argon2) throw new Error('Argon2 not loaded (argon2-bundled.min.js)');

        const { memory_kb, iterations, parallelism, hash_len } = params;
        if (!Number.isInteger(memory_kb) || memory_kb <= 0 || memory_kb > MAX_ARGON2_MEM_KB) throw new Error('Invalid Argon2 memory_kb');
        if (!Number.isInteger(iterations) || iterations <= 0 || iterations > MAX_ARGON2_ITERS) throw new Error('Invalid Argon2 iterations');
        if (!Number.isInteger(parallelism) || parallelism <= 0 || parallelism > MAX_ARGON2_PAR) throw new Error('Invalid Argon2 parallelism');
        if (![16, 24, 32].includes(hash_len)) throw new Error('Argon2 hash_len must be 16, 24, or 32');

        const options = {
            pass: password,
            salt: saltBytes,
            time: iterations,
            mem: memory_kb,
            parallelism,
            hashLen: hash_len,
            type: self.argon2.ArgonType.Argon2id
        };

        const started = Date.now();
        const res = await self.argon2.hash(options);
        const elapsed = Date.now() - started;

        const out = (res && res.hash) ? res.hash : new Uint8Array(AS.fromHex(res.hashHex));
        if (out.length !== hash_len) throw new Error('Unexpected Argon2 hash length');

        _lastKdfReport = {
            requested: { memory_kb, iterations, parallelism, hash_len },
            observed: {
                time_ms: (res && res.time) || elapsed,
                memory_kb_observed: (res && res.memory) || memory_kb,
                version: res && res.version
            }
        };
        return out;
    };

    const POLY = 0x11b;

    const gfAdd = (a, b) => a ^ b;

    const gfMul = (a0, b0) => {
        let a = a0 & 0xff, b = b0 & 0xff, p = 0;
        for (let i = 0; i < 8; i++) {
            if (b & 1) p ^= a;
            const hi = a & 0x80;
            a = (a << 1) & 0xff;
            if (hi) a ^= (POLY & 0xff);
            b >>= 1;
        }
        return p & 0xff;
    };

    const gfPow = (a, e) => {
        let r = 1, base = a & 0xff, exp = e >>> 0;
        while (exp > 0) {
            if (exp & 1) r = gfMul(r, base);
            base = gfMul(base, base);
            exp >>>= 1;
        }
        return r & 0xff;
    };

    const gfInv = (a) => {
        if (a === 0) throw new Error('No inverse for 0');
        return gfPow(a, 254);
    };

    const gfPolyEval = (coeffs, x) => {
        let y = 0;
        for (let i = coeffs.length - 1; i >= 0; i--) { y = gfMul(y, x); y = gfAdd(y, coeffs[i]); }
        return y & 0xff;
    };

    const gfLagrangeAtZero = (points) => {
        const k = points.length;
        let secret = 0;
        for (let i = 0; i < k; i++) {
            const xi = points[i].x & 0xff, yi = points[i].y & 0xff;
            let num = 1, den = 1;
            for (let m = 0; m < k; m++) {
                if (m === i) continue;
                const xm = points[m].x & 0xff;
                num = gfMul(num, xm);
                const diff = gfAdd(xm, xi);
                if (diff === 0) throw new Error('Duplicate share IDs');
                den = gfMul(den, diff);
            }
            secret = gfAdd(secret, gfMul(yi, gfMul(num, gfInv(den))));
        }
        return secret & 0xff;
    };

    const shamirSplit = (secretBytes, N, T) => {
        if (!(secretBytes instanceof Uint8Array)) throw new Error('secretBytes must be Uint8Array');
        if (!Number.isInteger(N) || N < 2 || N > 255) throw new Error('N must be 2..255');
        if (!Number.isInteger(T) || T < 2 || T > N) throw new Error('Threshold must be 2..N');

        const len = secretBytes.length;
        const polys = Array(len);

        for (let i = 0; i < len; i++) {
            const coeffs = new Uint8Array(T);
            coeffs[0] = secretBytes[i];
            if (T > 1) {
                const rnd = randBytes(T - 1);
                for (let j = 1; j < T; j++) coeffs[j] = rnd[j - 1] || 0;
                wipe(rnd);
            }
            polys[i] = coeffs;
        }

        const shares = [];
        for (let id = 1; id <= N; id++) {
            const y = new Uint8Array(len);
            for (let b = 0; b < len; b++) y[b] = gfPolyEval(polys[b], id);
            shares.push({ id, bytes: y });
        }
        for (let i = 0; i < len; i++) wipe(polys[i]);
        return shares;
    };

    const shamirCombine = (shares, T, len) => {
        if (!Array.isArray(shares) || shares.length < T) throw new Error('Not enough shares');
        const use = shares.slice(0, T);
        const secret = new Uint8Array(len);
        for (let b = 0; b < len; b++) {
            const pts = use.map((s) => ({ x: s.id, y: s.bytes[b] }));
            secret[b] = gfLagrangeAtZero(pts);
        }
        return secret;
    };

    const nowIso = () => {
        const d = new Date();
        d.setSeconds(0, 0);
        return d.toISOString();
    };

    const jsonToHex = (obj) => {
        const j = JSON.stringify(obj);
        const bytes = AS.encodeText(j);
        const hex = AS.toHex(bytes);
        wipe(bytes);
        return hex;
    };

    const hexToJson = (hex) => {
        const bytes = new Uint8Array(AS.fromHex(hex));
        const j = AS.decodeText(bytes);
        wipe(bytes);
        return JSON.parse(j);
    };

    const kdfStr   = (k)      => `argon2id|mem=${k.memory_kb}|it=${k.iterations}|par=${k.parallelism}|len=${k.hash_len}`;
    const masterAAD = (k)     => AS.encodeText(`master|${kdfStr(k)}`);
    const slaveAAD  = (k, id) => AS.encodeText(`slave|id=${id}|${kdfStr(k)}`);
    const masterSlavesAAD = (k) => AS.encodeText(`master_slaves|${kdfStr(k)}`);

    const buildCompactSpec = (kdfParams) => {
        return {
            kdfStr: "{kdf.alg}|mem={kdf.memory_kb}|it={kdf.iterations}|par={kdf.parallelism}|len={kdf.hash_len}",
            aad: {
                master: "master|{kdfStr}",
                slave: "slave|id={id}|{kdfStr}",
                master_slaves: "master_slaves|{kdfStr}"
            },
            shamir: {
                field: "GF(2^8), irreducible poly 0x11b",
                combine: "bytewise Lagrange interpolation at x=0; share IDs are integers 1..N"
            },
            mac: {
                alg: "HMAC-SHA256",
                key: "SHA-256(CEK || 'shamircipher-meta-mac-key')",
                msg: "canonical JSON of payload without 'mac' (UTF-8), keys sorted lexicographically at all levels; string values MAC’d as raw UTF-8 with no Unicode normalization (no NFC/NFD/compat)"
            },
            steps: "Recover CEK via master or T-of-N slaves; verify MAC as above; decrypt content using payload.cipher with content.iv (no AAD)."
        };
    };

    const sha256HexOfUtf8 = async (s) => {
        const b = AS.encodeText(s);
        const h = await sha256(b);
        const hex = AS.toHex(h);
        wipe(b); wipe(h);
        return hex;
    };

    const encryptWithMasterAndSlaves = async ({ plaintext, masterPassword, slavePasswords, kdfParams, threshold, name, description, masterHint }) => {
        if (typeof plaintext !== 'string') throw new Error('plaintext must be string');
        if (!masterPassword) throw new Error('Master password required');

        const N = Array.isArray(slavePasswords) ? slavePasswords.length : 0;
        if (N === 1) throw new Error('If using slaves, set N ≥ 2. Otherwise set N = 0 for master-only.');
        const T = Number.isInteger(threshold) ? threshold : (N > 0 ? Math.max(2, Math.ceil(N / 2)) : 0);
        if (N === 0 && T !== 0) throw new Error('For master-only mode, set threshold T = 0.');
        if (N >= 2 && (T < 2 || T > N)) throw new Error('Threshold must be between 2 and N when N ≥ 2.');

        const cek = randBytes(32);
        const contentIv = randBytes(12);
        const contentCt = await aesGcmEncrypt(cek, contentIv, AS.encodeText(plaintext));

        const mSalt = randBytes(16);
        const mKey  = await deriveKeyArgon2id(masterPassword, mSalt, kdfParams);
        const mIv   = randBytes(12);
        const mAAD  = masterAAD(kdfParams);
        const wrappedCek = await aesGcmEncrypt(mKey, mIv, cek, mAAD);
        wipe(mKey); wipe(mAAD);

        let shares_meta = [];
        let slaves = [];
        let master_slaves = null;

        if (N >= 2) {
            const shares = shamirSplit(cek, N, T);
            for (let i = 0; i < N; i++) {
                const s = shares[i];
                const salt = randBytes(16);
                const key  = await deriveKeyArgon2id(slavePasswords[i], salt, kdfParams);
                const iv   = randBytes(12);
                const aad  = slaveAAD(kdfParams, s.id);
                const ct   = await aesGcmEncrypt(key, iv, s.bytes, aad);
                wipe(key); wipe(s.bytes); wipe(aad);
                slaves.push({ id: s.id, salt: AS.toHex(salt), iv: AS.toHex(iv), ciphertext: AS.toHex(ct) });
                wipe(salt); wipe(iv); wipe(ct);
            }
            shares_meta = shares.map((s) => ({ id: s.id }));

            const msSalt = randBytes(16);
            const msKey  = await deriveKeyArgon2id(masterPassword, msSalt, kdfParams);
            const msIv   = randBytes(12);
            const msAAD  = masterSlavesAAD(kdfParams);
            const msPlain = AS.encodeText(JSON.stringify(slavePasswords));
            const msCt   = await aesGcmEncrypt(msKey, msIv, msPlain, msAAD);
            master_slaves = { salt: AS.toHex(msSalt), iv: AS.toHex(msIv), ciphertext: AS.toHex(msCt), aad_tag: `master_slaves|${kdfStr(kdfParams)}` };
            wipe(msKey); wipe(msAAD); wipe(msPlain); wipe(msSalt); wipe(msIv); wipe(msCt);
        }

        const payloadNoMac = {
            version: 1,
            cipher: 'AES-GCM-256-NoPadding',
            cek_length: 32,
            kdf: {
                alg: 'argon2id',
                memory_kb: kdfParams.memory_kb,
                iterations: kdfParams.iterations,
                parallelism: kdfParams.parallelism,
                hash_len: kdfParams.hash_len
            },
            master: { salt: AS.toHex(mSalt), iv: AS.toHex(mIv), wrapped_cek: AS.toHex(wrappedCek) },
            shamir: { n: N, threshold: T, shares_meta },
            slaves,
            content: { iv: AS.toHex(contentIv), ciphertext: AS.toHex(contentCt) },
            created_at: nowIso(),
            guide: buildCompactSpec(kdfParams)
        };

        if (typeof name === 'string' && name.length > 0) payloadNoMac.name = name;
        if (typeof description === 'string' && description.length > 0) payloadNoMac.description = description;
        if (typeof masterHint === 'string' && masterHint.length > 0) payloadNoMac.master_hint = masterHint;
        if (master_slaves) payloadNoMac.master_slaves = master_slaves;

        const mac = await computePayloadMac(cek, payloadNoMac);
        payloadNoMac.mac = AS.toHex(mac);

        wipe(cek); wipe(mSalt); wipe(mIv); wipe(wrappedCek);
        wipe(contentIv); wipe(contentCt); wipe(mac);

        const innerJson = JSON.stringify(payloadNoMac);
        const checksum = await sha256HexOfUtf8(innerJson);
        const outer = {
            payload: innerJson,
            checksum_alg: "sha256(payload)",
            checksum: checksum
        };

        return jsonToHex(outer);
    };

    const tryMasterPath = async (obj, masterPassword) => {
        if (!masterPassword) throw new Error('Master password not provided');
        const { kdf, master } = obj;
        const salt    = new Uint8Array(AS.fromHex(master.salt));
        const iv      = new Uint8Array(AS.fromHex(master.iv));
        const wrapped = new Uint8Array(AS.fromHex(master.wrapped_cek));
        const aad     = masterAAD(kdf);
        const key     = await deriveKeyArgon2id(masterPassword, salt, kdf);
        try { return await aesGcmDecrypt(key, iv, wrapped, aad); }
        finally { wipe(key); wipe(salt); wipe(iv); wipe(wrapped); wipe(aad); }
    };

    const trySlavePath = async (obj, provided) => {
        const { kdf, shamir, slaves } = obj;
        const T   = shamir.threshold;
        if (!Number.isInteger(T) || T <= 0) throw new Error('This payload has no slave shares.');
        const len = obj.cek_length || 32;

        const unused = new Map();
        for (const rec of slaves) unused.set(rec.id, rec);

        const collected = [];

        const decryptShare = async (rec, password) => {
            const salt = new Uint8Array(AS.fromHex(rec.salt));
            const iv   = new Uint8Array(AS.fromHex(rec.iv));
            const ct   = new Uint8Array(AS.fromHex(rec.ciphertext));
            const key  = await deriveKeyArgon2id(password, salt, kdf);
            const aad  = slaveAAD(kdf, rec.id);
            try {
                const bytes = await aesGcmDecrypt(key, iv, ct, aad);
                if (bytes.length !== len) throw new Error('Share length mismatch');
                return bytes;
            } finally { wipe(key); wipe(salt); wipe(iv); wipe(ct); wipe(aad); }
        };

        for (const pk of provided) {
            if (!pk || !pk.password) continue;
            for (const [id, rec] of Array.from(unused.entries())) {
                try {
                    const bytes = await decryptShare(rec, pk.password);
                    collected.push({ id: rec.id, bytes });
                    unused.delete(rec.id);
                    break;
                } catch {}
            }
            if (collected.length >= T) break;
        }

        if (collected.length < T) throw new Error(`Not enough of correct slaves (I have ${collected.length}, but I need ${T})`);
        const cek = shamirCombine(collected, T, len);
        for (const c of collected) wipe(c.bytes);
        return cek;
    };

    const parseAndVerifyOuter = async (hex) => {
        const outer = hexToJson(hex);
        if (!outer || typeof outer !== 'object' || typeof outer.payload !== 'string') {
            throw new Error('Invalid payload wrapper.');
        }
        const algo = String(outer.checksum_alg || '');
        if (algo !== 'sha256(payload)') throw new Error('Unsupported checksum algorithm.');
        if (typeof outer.checksum !== 'string' || !/^[0-9a-f]{64}$/i.test(outer.checksum)) {
            throw new Error('Invalid checksum.');
        }
        const computed = await sha256HexOfUtf8(outer.payload);
        const ok = computed.toLowerCase() === outer.checksum.toLowerCase();
        if (!ok) throw new Error('Checksum verification failed.');
        const inner = JSON.parse(outer.payload);
        return inner;
    };

    const decryptPayloadHex = async (hex, { masterPassword, slaveKeys }) => {
        const obj = await parseAndVerifyOuter(hex);
        if (obj.version !== 1) throw new Error(`Unsupported payload version: ${obj.version}`);

        const contentIv = new Uint8Array(AS.fromHex(obj.content.iv));
        const contentCt = new Uint8Array(AS.fromHex(obj.content.ciphertext));

        let cek = null;
        let via = null;

        const T = Number.isInteger(obj?.shamir?.threshold) ? obj.shamir.threshold : 0;

        if (masterPassword) {
            try { cek = await tryMasterPath(obj, masterPassword); via = 'master'; } catch {}
        }
        if (!cek) {
            if (T > 0) {
                if (!Array.isArray(slaveKeys) || slaveKeys.length === 0) throw new Error('Provide a correct master password or correct slave passwords');
                cek = await trySlavePath(obj, slaveKeys);
                via = 'slaves';
            } else {
                throw new Error('This payload was created in master-only mode. Provide the master password.');
            }
        }

        if (!obj.mac) { wipe(cek); throw new Error('Payload missing MAC'); }
        const ok = await verifyPayloadMac(cek, (() => { const c = { ...obj }; delete c.mac; return c; })(), obj.mac);
        if (!ok) { wipe(cek); throw new Error('Payload MAC verification failed'); }

        let revealedSlavePasswords = undefined;

        try {
            const ptBytes = await aesGcmDecrypt(cek, contentIv, contentCt);
            const pt = AS.decodeText(ptBytes);
            wipe(ptBytes);

            if (via === 'master' && obj.master_slaves && masterPassword) {
                try {
                    const ms = obj.master_slaves;
                    const msSalt = new Uint8Array(AS.fromHex(ms.salt));
                    const msIv   = new Uint8Array(AS.fromHex(ms.iv));
                    const msCt   = new Uint8Array(AS.fromHex(ms.ciphertext));
                    const msAAD  = masterSlavesAAD(obj.kdf);
                    const msKey  = await deriveKeyArgon2id(masterPassword, msSalt, obj.kdf);
                    try {
                        const msPt   = await aesGcmDecrypt(msKey, msIv, msCt, msAAD);
                        const json   = AS.decodeText(msPt);
                        wipe(msPt);
                        try {
                            const arr = JSON.parse(json);
                            if (AS.isArrayOfStrings && AS.isArrayOfStrings(arr) && arr.length > 0) {
                                revealedSlavePasswords = arr.slice();
                            }
                        } catch {}
                    } finally { wipe(msKey); wipe(msSalt); wipe(msIv); wipe(msCt); wipe(msAAD); }
                } catch {}
            }

            return { plaintext: pt, via, meta: obj, ...(revealedSlavePasswords ? { revealedSlavePasswords } : {}) };
        } finally { wipe(cek); wipe(contentIv); wipe(contentCt); }
    };

    self.ShamirCipher = {
        toHex: AS.toHex,
        fromHex: AS.fromHex,
        utf8ToBytes: AS.encodeText,
        bytesToUtf8: AS.decodeText,
        encryptWithMasterAndSlaves,
        decryptPayloadHex,
        shamirSplit, shamirCombine,
        canonicalStringify,
        getLastKdfReport
    };
})();

(() => {
    'use strict';

    const AS = self.AppShared;

    const $    = (sel) => document.querySelector(sel);
    const byId = (id)  => document.getElementById(id);
    const clearNode = (node) => { if (node) node.innerHTML = ''; };

    const bindCopy = (btnSel, srcSel) => {
        const btn = $(btnSel), src = $(srcSel);
        if (!btn || !src) return;
        btn.addEventListener('click', () => {
            const text = src.value ?? src.innerText ?? '';
            AS.copyToClipboard(text, 'Copied');
        });
    };

    const applyRngWarningVisibility = () => {
        const warn = byId('rngWarning');
        if (!warn) return;
        const ok = AS.hasWebCrypto();
        warn.classList.toggle('is-hidden', !!ok);
    };

    const bindModeSwitching = () => {
        const btnEnc   = byId('modeEncrypt');
        const btnDec   = byId('modeDecrypt');
        const panelEnc = byId('panelEncrypt');
        const panelDec = byId('panelDecrypt');
        if (!btnEnc || !btnDec || !panelEnc || !panelDec) return;

        const setMode = (encMode) => {
            panelEnc.classList.toggle('is-hidden', !encMode);
            panelDec.classList.toggle('is-hidden', encMode);
            btnEnc.classList.toggle('is-active', encMode);
            btnDec.classList.toggle('is-active', !encMode);
            btnEnc.setAttribute('aria-selected', String(encMode));
            btnDec.setAttribute('aria-selected', String(!encMode));
            if (!encMode) syncDecPayloadHeight();
        };

        btnEnc.addEventListener('click', () => setMode(true));
        btnDec.addEventListener('click', () => setMode(false));
        setMode(true);
    };

    const styleInputForInlineBtn = (input) => {
        if (!input) return;
        input.style.minHeight = '44px';
        input.style.paddingRight = '3rem';
    };

    const attachGenerateHandler = (btn, input, label) => {
        if (!btn || !input) return;
        btn.addEventListener('click', () => {
            const res = AS.generateRandomString({ label: label || 'VALUE' });
            input.value = res.value || '';
            if (res.ok) AS.showToast('Random password generated');
            else AS.showModal('Randomness warning', res.message);
        });
    };

    const buildSlavePasswordFields = (n, container, thresholdInput) => {
        const N = Math.max(0, Math.min(255, Number.isFinite(n) ? n : 0));
        container.innerHTML = '';
        for (let i = 1; i <= N; i++) {
            const field = document.createElement('div');
            field.className = 'field';
            const label = document.createElement('label');
            label.setAttribute('for', `enc_slave_${i}`);
            label.textContent = `Slave #${i}`;
            const body = document.createElement('div');
            body.className = 'field-body';
            const inp = document.createElement('input');
            inp.id = `enc_slave_${i}`;
            inp.type = 'text';
            inp.placeholder = 'Slave password';
            inp.className = 'enc-slave-pw';
            styleInputForInlineBtn(inp);
            const gen = document.createElement('button');
            gen.type = 'button';
            gen.className = 'copy-btn';
            gen.setAttribute('aria-label', `Generate random for Slave #${i}`);
            gen.textContent = 'Generate';
            body.appendChild(inp);
            body.appendChild(gen);
            attachGenerateHandler(gen, inp, 'PASSWORD');
            field.append(label, body);
            container.appendChild(field);
        }
        const t = Math.max(0, Math.min(N, parseInt(thresholdInput.value || '0', 10)));
        thresholdInput.value = String(t);
        return N;
    };

    const clampIterations = (v) => {
        const n = Number.isFinite(v) ? Math.trunc(v) : 1;
        return Math.max(4, Math.min(64, n));
    };

    const clampMemKb = (v) => {
        const n = Number.isFinite(v) ? Math.trunc(v) : 131072;
        const min = 131072;
        const max = 1048576;
        return Math.max(min, Math.min(max, n));
    };

    const readEncryptKdfParams = () => {
        const memRaw = parseInt(byId('enc_kdf_mem').value, 10);
        const itRaw  = parseInt(byId('enc_kdf_iter').value, 10);
        const mem = clampMemKb(memRaw);
        const it  = clampIterations(itRaw);
        if (mem !== memRaw) {
            byId('enc_kdf_mem').value = String(mem);
            if (mem < memRaw) AS.showToast('Lowered memory to maximum allowed 1048576 KB');
            else AS.showToast('Raised memory to minimum safe 131072 KB');
        }
        if (it  !== itRaw ) { byId('enc_kdf_iter').value = String(it); AS.showToast('Raised iterations to minimum safe 4'); }
        return { memory_kb: mem, iterations: it, parallelism: 1, hash_len: 32 };
    };

    const bindEncryptPanel = () => {
        const slavesContainer = byId('enc_slavesContainer');
        const nInput = byId('enc_nslaves');
        const tInput = byId('enc_threshold');
        const status = byId('enc_status');
        const btnGenerate = byId('enc_btnGenerate');
        const btnClear = byId('enc_btnClear');

        if (!slavesContainer || !nInput || !tInput) return;

        buildSlavePasswordFields(parseInt(nInput.value || '3', 10), slavesContainer, tInput);

        const clampT = () => {
            const n = Math.max(0, Math.min(255, parseInt(nInput.value || '0', 10)));
            nInput.value = String(n);
            const t = Math.max(0, Math.min(n, parseInt(tInput.value || '0', 10)));
            tInput.value = String(t);
        };

        const rebuild = () => { clampT(); buildSlavePasswordFields(parseInt(nInput.value, 10), slavesContainer, tInput); };
        nInput.addEventListener('input', rebuild);
        nInput.addEventListener('change', rebuild);

        const masterInput = byId('enc_masterpwd');
        const masterBtn = byId('enc_btnGenMaster');
        styleInputForInlineBtn(masterInput);
        attachGenerateHandler(masterBtn, masterInput, 'PASSWORD');

        btnClear && btnClear.addEventListener('click', () => {
            byId('enc_plaintext').value = '';
            byId('enc_masterpwd').value = '';
            byId('enc_master_hint').value = '';
            byId('enc_name').value = '';
            byId('enc_desc').value = '';
            Array.from(slavesContainer.querySelectorAll('.enc-slave-pw')).forEach((i) => (i.value = ''));
            byId('enc_result').value = '';
            if (status) status.textContent = '';
        });

        btnGenerate && btnGenerate.addEventListener('click', async () => {
            const controls = [btnGenerate, btnClear, byId('modeEncrypt'), byId('modeDecrypt')];
            await AS.runWithBusy(async () => {
                try {
                    clampT();
                    const plaintext = byId('enc_plaintext').value;
                    const name = (byId('enc_name').value || '').trim();
                    const description = (byId('enc_desc').value || '').trim();
                    const masterHint = (byId('enc_master_hint').value || '').trim();
                    if (name.length > 32) { AS.showModal('Invalid Name', 'Name must be at most 32 characters.'); return; }
                    if (description.length > 2048) { AS.showModal('Invalid Description', 'Description must be at most 2048 characters.'); return; }
                    if (masterHint.length > 256) { AS.showModal('Invalid Hint', 'Master password hint must be at most 256 characters.'); return; }
                    const masterpwd = byId('enc_masterpwd').value;
                    if (!masterpwd) { AS.showModal('Missing master password', 'Enter a master password to continue.'); return; }
                    const n = parseInt(nInput.value, 10);
                    const slavePwds = Array.from(slavesContainer.querySelectorAll('.enc-slave-pw')).map((i) => i.value);
                    if (slavePwds.length !== n) { AS.showModal('Invalid input', 'Mismatch in number of slave passwords.'); return; }
                    if (n > 0 && slavePwds.some((p) => !p || p.length === 0)) { AS.showModal('Invalid input', 'All slave passwords must be provided.'); return; }
                    if (n === 1) { AS.showModal('Invalid N', 'If using slaves, set N ≥ 2. Otherwise set N = 0 for master-only.'); return; }
                    const t = parseInt(tInput.value, 10);
                    if (!Number.isInteger(t) || t < 0 || t > n) { AS.showModal('Invalid threshold', 'Threshold T must be between 0 and N.'); return; }
                    if (n === 0 && t !== 0) { AS.showModal('Invalid threshold', 'For master-only mode set T = 0.'); return; }
                    if (n >= 2 && t < 2) { AS.showModal('Invalid threshold', 'When N ≥ 2, set T ≥ 2.'); return; }

                    const kdfParams = readEncryptKdfParams();
                    const hexPayload = await self.ShamirCipher.encryptWithMasterAndSlaves({
                        plaintext,
                        masterPassword: masterpwd,
                        slavePasswords: slavePwds,
                        kdfParams,
                        threshold: t,
                        name: name || undefined,
                        description: description || undefined,
                        masterHint: masterHint || undefined
                    });

                    byId('enc_result').value = hexPayload;
                    if (status) status.textContent = '';
                    AS.showModal('Encryption successful', 'Payload HEX created. Store it safely and clear data of this app.', 'success');
                } catch (e) {
                    console.error(e);
                    if (status) status.textContent = '';
                    AS.showModal('Error', (e && e.message) ? e.message : String(e));
                }
            }, { overlay: 'loadingOverlay', controls });
        });
    };

    const addDecryptSlaveRow = (container) => {
        const row = document.createElement('div');
        row.className = 'dec-slave-row';

        const inner = document.createElement('div');
        inner.className = 'grid';

        const pwWrap = document.createElement('div');
        pwWrap.className = 'small-field';
        const pwLbl = document.createElement('span');
        pwLbl.className = 'label-small';
        pwLbl.textContent = 'Slave password';
        const pw = document.createElement('input');
        pw.type = 'text';
        pw.placeholder = 'Password';
        pw.className = 'dec-slave-pw';
        pwWrap.append(pwLbl, pw);

        inner.append(pwWrap);
        row.appendChild(inner);
        container.appendChild(row);
    };

    const setText = (id, val) => { const el = byId(id); if (el) el.textContent = val ?? ''; };
    const setHidden = (idWrap, hidden) => { const el = byId(idWrap); if (el) el.classList.toggle('is-hidden', !!hidden); };

    const fmtIsoToLocal = (iso) => {
        if (!iso) return '';
        const d = new Date(iso);
        const pad = (n)=>String(n).padStart(2,'0');
        const yyyy = d.getFullYear();
        const MM = pad(d.getMonth()+1);
        const DD = pad(d.getDate());
        const hh = pad(d.getHours());
        const mm = pad(d.getMinutes());
        const ss = pad(d.getSeconds());
        return `${yyyy}-${MM}-${DD} ${hh}:${mm}:${ss}`;
    };

    const renderParsedInfo = (obj) => {
        const hintBtn = byId('dec_master_hint_btn');

        if (!obj) {
            ['pi_version','pi_created','pi_n','pi_t','pi_ids','pi_mem','pi_iter','pi_par','pi_hlen','pi_name','pi_desc'].forEach((k)=>setText(k,''));
            setHidden('pi_name_wrap', true);
            setHidden('pi_desc_wrap', true);
            if (hintBtn) { hintBtn.classList.add('is-hidden'); hintBtn.dataset.hint=''; }
            setText('dec_parse_status','');
            syncDecPayloadHeight();
            return;
        }

        const ids = (obj.slaves || []).map((s) => s.id).join(', ');
        const kdf = obj.kdf || {};

        if (typeof obj.name === 'string' && obj.name.length > 0) {
            setText('pi_name', obj.name);
            setHidden('pi_name_wrap', false);
        } else {
            setHidden('pi_name_wrap', true);
            setText('pi_name','');
        }

        if (typeof obj.description === 'string' && obj.description.length > 0) {
            setText('pi_desc', obj.description);
            setHidden('pi_desc_wrap', false);
        } else {
            setHidden('pi_desc_wrap', true);
            setText('pi_desc','');
        }

        if (hintBtn) {
            const hint = (typeof obj.master_hint === 'string' && obj.master_hint.length > 0) ? obj.master_hint : '';
            if (hint) {
                hintBtn.dataset.hint = hint;
                hintBtn.classList.remove('is-hidden');
            } else {
                hintBtn.dataset.hint = '';
                hintBtn.classList.add('is-hidden');
            }
        }

        setText('pi_mem', String(kdf.memory_kb ?? ''));
        setText('pi_iter', String(kdf.iterations ?? ''));
        setText('pi_par', String(kdf.parallelism ?? ''));
        setText('pi_hlen', String(kdf.hash_len ?? ''));

        setText('pi_n', String(obj.shamir?.n ?? ''));
        setText('pi_ids', ids);
        setText('pi_t', String(obj.shamir?.threshold ?? ''));

        setText('pi_created', fmtIsoToLocal(obj.created_at || ''));
        setText('pi_version', String(obj.version ?? ''));
        setText('dec_parse_status','');
        syncDecPayloadHeight();
    };

    const TEN_MIB = 10 * 1024 * 1024;
    const cleanHexAndLen = (hex) => {
        const clean = (hex || '').replace(/[^0-9a-f]/gi, '').toLowerCase();
        const byteLen = Math.floor(clean.length / 2);
        return { clean, byteLen };
    };

    // Parse outer wrapper and verify checksum for preview if possible.
    const tryParsePayload = (hex) => {
        if (!hex || hex.trim() === '') return null;
        const { clean, byteLen } = cleanHexAndLen(hex);
        if (byteLen > TEN_MIB) return 'too_big';
        try {
            const outerBytes = self.ShamirCipher.fromHex(clean);
            const outerStr = new TextDecoder().decode(outerBytes);
            const outer = JSON.parse(outerStr);
            if (!outer || typeof outer !== 'object' || typeof outer.payload !== 'string') return 'error';
            if (String(outer.checksum_alg || '') !== 'sha256(payload)') return 'error';
            if (typeof outer.checksum !== 'string' || !/^[0-9a-f]{64}$/i.test(outer.checksum)) return 'error';
            // compute checksum
            const enc = new TextEncoder();
            const payloadBytes = enc.encode(outer.payload);
            return crypto.subtle.digest('SHA-256', payloadBytes).then((d)=>{
                const got = Array.from(new Uint8Array(d)).map(b=>b.toString(16).padStart(2,'0')).join('');
                if (got.toLowerCase() !== outer.checksum.toLowerCase()) return 'checksum_fail';
                try {
                    const inner = JSON.parse(outer.payload);
                    return inner && typeof inner === 'object' ? inner : 'error';
                } catch { return 'error'; }
            });
        } catch { return 'error'; }
    };

    const syncDecPayloadHeight = () => {
        const ta = byId('dec_payload');
        const card = byId('dec_infoCard');
        if (!ta || !card) return;

        const twoCols = window.matchMedia('(min-width: 951px)').matches;

        ta.style.height = '';
        card.style.height = '';

        if (!twoCols) { ta.style.maxHeight = ''; return; }

        ta.style.maxHeight = 'none';

        const taH = ta.getBoundingClientRect().height || 0;
        const cardH = card.getBoundingClientRect().height || 0;
        const target = Math.max(taH, cardH);

        if (target > 0) {
            ta.style.height = `${target}px`;
            card.style.height = `${target}px`;
        }
    };

    const bindDecryptPanel = () => {
        const slaveKeysDiv  = byId('dec_slaveKeys');
        const status        = byId('dec_status');
        const btnAdd        = byId('dec_btnAddSlave');
        const btnClearRows  = byId('dec_btnClearSlaves');
        const btnDecrypt    = byId('dec_btnDecrypt');
        const btnClearAll   = byId('dec_btnClearAll');
        const payloadTa     = byId('dec_payload');
        const parseStatus   = byId('dec_parse_status');
        const revealCard    = byId('dec_revealCard');
        const revealTa      = byId('dec_revealed_slaves');
        const masterPwInput = byId('dec_masterpwd');
        const hintBtn       = byId('dec_master_hint_btn');

        if (!slaveKeysDiv) return;

        if (masterPwInput) {
            masterPwInput.style.minHeight = '44px';
            masterPwInput.style.paddingRight = '3rem';
        }
        if (hintBtn) {
            hintBtn.addEventListener('click', () => {
                const hint = hintBtn.dataset.hint || '';
                if (hint) AS.showModal('Master password hint', hint);
            });
        }

        btnAdd && btnAdd.addEventListener('click', () => { addDecryptSlaveRow(slaveKeysDiv); });
        btnClearRows && btnClearRows.addEventListener('click', () => { clearNode(slaveKeysDiv); });

        const handlePayloadInput = async () => {
            const hex = payloadTa.value;
            const parsed = await tryParsePayload(hex);
            if (parsed && parsed !== 'error' && parsed !== 'too_big' && parsed !== 'checksum_fail') {
                renderParsedInfo(parsed);

                const n = Number.isInteger(parsed?.shamir?.n) ? parsed.shamir.n : 0;
                const t = Number.isInteger(parsed?.shamir?.threshold) ? parsed.shamir.threshold : 0;

                if (n <= 0 || t <= 0) {
                    clearNode(slaveKeysDiv);
                } else {
                    const current = slaveKeysDiv.querySelectorAll('.dec-slave-row').length;
                    if (current !== t) {
                        clearNode(slaveKeysDiv);
                        for (let i = 0; i < t; i++) addDecryptSlaveRow(slaveKeysDiv);
                    }
                }

                if (parseStatus) parseStatus.textContent = '';
                syncDecPayloadHeight();
            } else if (parsed === 'checksum_fail') {
                renderParsedInfo(null);
                if (parseStatus) parseStatus.textContent = 'Checksum verification failed.';
                syncDecPayloadHeight();
            } else if (parsed === 'error') {
                renderParsedInfo(null);
                if (parseStatus) parseStatus.textContent = 'Invalid HEX or payload format.';
                syncDecPayloadHeight();
            } else if (parsed === 'too_big') {
                renderParsedInfo(null);
                if (parseStatus) parseStatus.textContent = 'Payload HEX is larger than 10 MiB; not parsing for safety.';
                syncDecPayloadHeight();
            } else {
                renderParsedInfo(null);
                if (parseStatus) parseStatus.textContent = '';
                syncDecPayloadHeight();
            }
        };

        payloadTa && payloadTa.addEventListener('input', handlePayloadInput);
        payloadTa && payloadTa.addEventListener('change', handlePayloadInput);
        payloadTa && payloadTa.addEventListener('paste', () => setTimeout(handlePayloadInput, 0));

        btnDecrypt && btnDecrypt.addEventListener('click', async () => {
            const controls = [btnDecrypt, btnAdd, btnClearRows, byId('modeEncrypt'), byId('modeDecrypt')];
            await AS.runWithBusy(async () => {
                try {
                    const hex = byId('dec_payload').value.trim();
                    if (!hex) { AS.showModal('Missing payload', 'Paste a hex-encoded payload first.'); return; }

                    const { clean, byteLen } = cleanHexAndLen(hex);
                    if (byteLen > TEN_MIB) { AS.showModal('Payload too large', 'Payload HEX is larger than 10 MiB; aborting for safety.'); return; }

                    const pre = await tryParsePayload(hex);
                    if (pre && pre !== 'error' && pre !== 'too_big' && pre !== 'checksum_fail') {
                        const mem = Number(pre?.kdf?.memory_kb || 0);
                        const it  = Number(pre?.kdf?.iterations || 0);
                        if (mem > 131072 || it > 4) {
                            AS.showToast('Warning: high KDF settings detected; decryption may take a while.');
                        }
                    } else if (pre === 'checksum_fail') {
                        AS.showModal('Checksum failed', 'The payload checksum does not match. Abort for safety.');
                        return;
                    }

                    const masterpwd = byId('dec_masterpwd').value;

                    const slaveRows = Array.from(slaveKeysDiv.querySelectorAll('.dec-slave-row'));
                    const slaveKeys = slaveRows.map((row) => {
                        const pwEl = row.querySelector('.dec-slave-pw');
                        const pw = pwEl ? pwEl.value : '';
                        return { password: pw };
                    }).filter((k) => k.password && k.password.length > 0);

                    const { plaintext, via, revealedSlavePasswords, meta } = await self.ShamirCipher.decryptPayloadHex(clean, {
                        masterPassword: masterpwd,
                        slaveKeys
                    });

                    byId('dec_plaintext').value = plaintext;

                    if (via === 'master' && Array.isArray(revealedSlavePasswords) && revealedSlavePasswords.length > 0) {
                        let lines;
                        if (meta && Array.isArray(meta.slaves)) {
                            lines = revealedSlavePasswords.map((pw, idx) => {
                                const rec = meta.slaves[idx];
                                const id = rec && typeof rec.id === 'number' ? rec.id : (idx + 1);
                                return `Share ID ${id}: ${pw}`;
                            });
                        } else {
                            lines = revealedSlavePasswords.map((pw, idx) => `Share ID ${idx+1}: ${pw}`);
                        }
                        if (revealTa) revealTa.value = lines.join('\n');
                        if (revealCard) revealCard.classList.remove('is-hidden');
                    } else {
                        if (revealTa) revealTa.value = '';
                        if (revealCard) revealCard.classList.add('is-hidden');
                    }

                    if (status) status.textContent = '';
                    AS.showModal('Decryption successful', `Decrypted using ${via === 'master' ? 'MASTER' : 'SLAVES'}.`, 'success');
                } catch (e) {
                    console.error(e);
                    byId('dec_plaintext').value = '';
                    const revealCard = byId('dec_revealCard'); const revealTa = byId('dec_revealed_slaves');
                    if (revealTa) revealTa.value = '';
                    if (revealCard) revealCard.classList.add('is-hidden');
                    if (status) status.textContent = '';
                    AS.showModal('Decryption failed', (e && e.message) ? e.message : String(e));
                }
            }, { overlay: 'loadingOverlay', controls });
        });

        btnClearAll && btnClearAll.addEventListener('click', () => {
            byId('dec_payload').value = '';
            byId('dec_masterpwd').value = '';
            byId('dec_plaintext').value = '';
            clearNode(slaveKeysDiv);
            renderParsedInfo(null);
            if (parseStatus) parseStatus.textContent = '';
            const revealCard = byId('dec_revealCard'); const revealTa = byId('dec_revealed_slaves');
            if (revealTa) revealTa.value = '';
            if (revealCard) revealCard.classList.add('is-hidden');
            if (status) status.textContent = '';
            syncDecPayloadHeight();
        });

        window.addEventListener('resize', syncDecPayloadHeight);
        syncDecPayloadHeight();
    };

    const init = () => {
        applyRngWarningVisibility();
        bindModeSwitching();
        bindCopy('#copyEncPlain', '#enc_plaintext');
        bindCopy('#copyEncResult', '#enc_result');
        bindCopy('#copyDecPayload', '#dec_payload');
        bindCopy('#copyDecPlain', '#dec_plaintext');
        bindCopy('#copyDecSlaves', '#dec_revealed_slaves');
        bindEncryptPanel();
        bindDecryptPanel();
    };

    if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', init);
    else init();
})();