(function (global) {
  "use strict";

  var d = document;

  // ---------- DOM ----------
  function $(id) { return d.getElementById(id); }

  // ---------- toast / modal helpers ----------
  function getToastEl() { return $("toast"); }
  function getModalEls() {
    var backdrop = $("modalBackdrop");
    return {
      backdrop: backdrop,
      modal: backdrop ? backdrop.querySelector(".modal") : null,
      title: $("modalTitle"),
      body: $("modalBody")
    };
  }
  function getOverlay(elOrId) {
    if (elOrId && elOrId instanceof HTMLElement) return elOrId;
    if (typeof elOrId === "string") return $(elOrId);
    return $("loadingOverlay");
  }
  function toElementArray(list) {
    if (!list) return [];
    return list.map(function (item) {
      if (!item) return null;
      if (item instanceof HTMLElement) return item;
      if (typeof item === "string") return $(item);
      return null;
    }).filter(Boolean);
  }

  function showToast(msg) {
    var toast = getToastEl();
    if (!toast) return;
    toast.textContent = msg;
    toast.classList.add("show");
    setTimeout(function () { toast.classList.remove("show"); }, 2600);
  }

  function showModal(title, message, variant) {
    var m = getModalEls();
    if (!m.backdrop) return;
    if (m.modal) {
      m.modal.classList.remove("success");
      if (variant === "success") m.modal.classList.add("success");
    }
    if (m.title) m.title.textContent = title;
    if (m.body) m.body.textContent = message;
    m.backdrop.style.display = "flex";
  }

  function hideModal() {
    var m = getModalEls();
    if (m.backdrop) m.backdrop.style.display = "none";
  }

  // ---------- busy / overlay ----------
  function restartSpinner(overlayEl) {
    var overlay = getOverlay(overlayEl);
    if (!overlay) return;
    var spinner = overlay.querySelector(".spinner");
    if (!spinner) return;
    var fresh = spinner.cloneNode(true);
    spinner.replaceWith(fresh);
  }
  function setControlsDisabled(controls, disabled) {
    controls.forEach(function (el) { if (el) el.disabled = disabled; });
  }
  async function runWithBusy(fn, options) {
    options = options || {};
    var overlay = getOverlay(options.overlay);
    var controls = toElementArray(options.controls || []);
    setControlsDisabled(controls, true);
    if (overlay) {
      overlay.style.display = "flex";
      restartSpinner(overlay);
      await new Promise(function (r) { requestAnimationFrame(r); });
      await new Promise(function (r) { requestAnimationFrame(r); });
    }
    try {
      await fn();
    } finally {
      setControlsDisabled(controls, false);
      if (overlay) overlay.style.display = "none";
    }
  }

  // ---------- clipboard ----------
  function fallbackCopy(text, onSuccess, onError) {
    try {
      var ta = d.createElement("textarea");
      ta.value = text;
      ta.style.position = "fixed";
      ta.style.top = "-9999px";
      d.body.appendChild(ta);
      ta.focus(); ta.select();
      var ok = d.execCommand("copy");
      d.body.removeChild(ta);
      ok ? (onSuccess && onSuccess()) : (onError && onError());
    } catch (err) {
      if (onError) onError(err);
    }
  }
  function copyToClipboard(text, successMsg) {
    if (!text) { showToast("Nothing to copy"); return; }
    if (navigator.clipboard && window.isSecureContext) {
      navigator.clipboard.writeText(text).then(function () {
        showToast(successMsg || "Copied");
      }).catch(function () {
        fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                          function () { showToast("Copy failed"); });
      });
    } else {
      fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                        function () { showToast("Copy failed"); });
    }
  }

  // ---------- secure wipe ----------
  function secureWipe() {
    for (var i = 0; i < arguments.length; i++) {
      var it = arguments[i];
      if (!it) continue;
      if (it instanceof Uint8Array) it.fill(0);
      else if (it instanceof ArrayBuffer) new Uint8Array(it).fill(0);
    }
  }

  // ---------- randomness ----------
  function hasWebCrypto() {
    return typeof window !== "undefined" && window.crypto && typeof window.crypto.getRandomValues === "function";
  }
  function evaluateRandomResult(label, info) {
    var hasSecure = !!info.hasSecure;
    var mathRandomUsed = !!info.mathRandomUsed;
    if (!hasSecure) {
      return {
        ok: false, variant: null,
        message: "❌ THIS " + label + " IS NOT SECURE ❌ This browser did NOT provide crypto.getRandomValues(). Use a modern browser, offline."
      };
    }
    if (mathRandomUsed) {
      return {
        ok: false, variant: null,
        message: "❌ THIS " + label + " IS NOT SECURE ❌ The library fell back to Math.random(). Use another browser/build."
      };
    }
    return { ok: true, variant: "success", message: "✅ Secure " + label.toLowerCase() + " generated successfully." };
  }
  var DEFAULT_PASSWORD_CHARSET = "ABCDEFGHJKLMNPQRSTUVWXYZabcdefghkmnpqrstuvwxyz2346789";
  function generateRandomString(opts) {
    opts = opts || {};
    var length = opts.length || 32;
    var charset = opts.charset || DEFAULT_PASSWORD_CHARSET;
    var label   = opts.label   || "VALUE";

    var secure = hasWebCrypto();
    var buf = new Uint8Array(length);
    var mathRandomUsed = false;

    if (secure) {
      window.crypto.getRandomValues(buf);
    } else {
      mathRandomUsed = true;
      for (var i = 0; i < length; i++) buf[i] = Math.floor(Math.random() * 256);
    }

    var raw = "";
    var mod = charset.length;
    for (var j = 0; j < length; j++) raw += charset[buf[j] % mod];

    secureWipe(buf);

    var groups = [];
    for (var k = 0; k < raw.length; k += 4) groups.push(raw.slice(k, k + 4));
    var formatted = groups.join("-");

    var assessment = evaluateRandomResult(label, { hasSecure: secure, mathRandomUsed: mathRandomUsed });

    return { value: formatted, ok: assessment.ok, variant: assessment.variant, message: assessment.message };
  }

  // ---------- hex / bytes / text ----------
  function bytesToHex(bytes) {
    return Array.prototype.map.call(bytes, function (b) { return b.toString(16).padStart(2, "0"); }).join("");
  }
  function hexToBytes(hex) {
    if (!hex) return new Uint8Array(0);
    var clean = String(hex).replace(/[^0-9a-f]/gi, "").toLowerCase();
    if (clean.length % 2 !== 0) throw new Error("Hex length must be even.");
    var out = new Uint8Array(clean.length / 2);
    for (var i = 0; i < clean.length; i += 2) out[i / 2] = parseInt(clean.substr(i, 2), 16);
    return out;
  }
  function toHex(data) {
    if (data instanceof ArrayBuffer) return bytesToHex(new Uint8Array(data));
    if (data instanceof Uint8Array) return bytesToHex(data);
    return "";
  }
  function fromHex(hex) { return hexToBytes(hex).buffer; }

  var _encoder = new TextEncoder();
  var _decoder = new TextDecoder("utf-8");
  function encodeText(str) { return _encoder.encode(str); }
  function decodeText(buf) { return _decoder.decode(buf); }

  // ---------- export ----------
  var AppShared = {
    $: $,
    showToast: showToast,
    showModal: showModal,
    hideModal: hideModal,
    restartSpinner: restartSpinner,
    runWithBusy: runWithBusy,
    copyToClipboard: copyToClipboard,
    fallbackCopy: fallbackCopy,
    secureWipe: secureWipe,
    hasWebCrypto: hasWebCrypto,
    evaluateRandomResult: evaluateRandomResult,
    generateRandomString: generateRandomString,
    DEFAULT_PASSWORD_CHARSET: DEFAULT_PASSWORD_CHARSET,
    bytesToHex: bytesToHex,
    hexToBytes: hexToBytes,
    toHex: toHex,
    fromHex: fromHex,
    encodeText: encodeText,
    decodeText: decodeText
  };
  global.AppShared = AppShared;

  // global modal interaction
  d.addEventListener("click", function (e) {
    var m = getModalEls();
    if (e.target && e.target.id === "modalCloseBtn") { hideModal(); return; }
    if (m.backdrop && e.target === m.backdrop) { hideModal(); }
  });
  d.addEventListener("keydown", function (e) {
    var m = getModalEls();
    if (e.key === "Escape" && m.backdrop && m.backdrop.style.display === "flex") hideModal();
  });
})(typeof window !== "undefined" ? window : this);

/* ---- RSATool app ---- */
(function () {
  "use strict";

  var S = window.AppShared;
  var $ = S.$;

  // tabs
  var tabReceiver = $("tabReceiver");
  var tabSender = $("tabSender");
  var receiverPanel = $("receiverPanel");
  var senderPanel = $("senderPanel");

  // receiver
  var generateKeyPairBtn = $("generateKeyPairBtn");
  var clearReceiverBtn = $("clearReceiverBtn");
  var receiverPublicKey = $("receiverPublicKey");
  var receiverPrivateKey = $("receiverPrivateKey");
  var copyReceiverPubBtn = $("copyReceiverPubBtn");
  var copyReceiverPrivBtn = $("copyReceiverPrivBtn");
  var ciphertextInput = $("ciphertextInput");
  var decryptBtn = $("decryptBtn");
  var decryptedOutput = $("decryptedOutput");
  var copyDecryptedBtn = $("copyDecryptedBtn");

  // sender
  var senderReceiverPubInput = $("senderReceiverPubInput");
  var useReceiverPubBtn = $("useReceiverPubBtn");
  var clearSenderBtn = $("clearSenderBtn");
  var senderWorkArea = $("senderWorkArea");
  var senderPassword = $("senderPassword");
  var generateSenderPasswordBtn = $("generateSenderPasswordBtn");
  var encryptBtn = $("encryptBtn");
  var encryptedOutput = $("encryptedOutput");
  var copyEncryptedOutputBtn = $("copyEncryptedOutputBtn");

  var receiverPrivateKeyObj = null;
  var senderPublicKeyObj = null;

  function switchToReceiver() {
    tabReceiver.classList.add("is-active");
    tabReceiver.setAttribute("aria-selected", "true");
    tabSender.classList.remove("is-active");
    tabSender.setAttribute("aria-selected", "false");
    receiverPanel.classList.remove("is-hidden");
    senderPanel.classList.add("is-hidden");
    senderPanel.setAttribute("aria-hidden", "true");
  }
  function switchToSender() {
    tabSender.classList.add("is-active");
    tabSender.setAttribute("aria-selected", "true");
    tabReceiver.classList.remove("is-active");
    tabReceiver.setAttribute("aria-selected", "false");
    senderPanel.classList.remove("is-hidden");
    senderPanel.setAttribute("aria-hidden", "false");
    receiverPanel.classList.add("is-hidden");
  }
  tabReceiver.addEventListener("click", switchToReceiver);
  tabSender.addEventListener("click", switchToSender);

  function clearReceiverFields() {
    receiverPublicKey.value = "";
    receiverPrivateKey.value = "";
    ciphertextInput.value = "";
    decryptedOutput.value = "";
    copyReceiverPubBtn.disabled = true;
    copyReceiverPrivBtn.disabled = true;
    copyDecryptedBtn.disabled = true;
    receiverPrivateKeyObj = null;
  }
  clearReceiverBtn.addEventListener("click", clearReceiverFields);

  async function generateReceiverKeys() {
    if (!S.hasWebCrypto()) {
      S.showModal("WebCrypto needed", "This browser does not expose window.crypto.subtle. Use a modern browser (offline).");
      return;
    }
    await S.runWithBusy(async function () {
      var keyPair = await window.crypto.subtle.generateKey(
        { name: "RSA-OAEP", modulusLength: 4096, publicExponent: new Uint8Array([1, 0, 1]), hash: "SHA-256" },
        true,
        ["encrypt", "decrypt"]
      );
      var spki = await window.crypto.subtle.exportKey("spki", keyPair.publicKey);
      var pkcs8 = await window.crypto.subtle.exportKey("pkcs8", keyPair.privateKey);

      receiverPublicKey.value = S.toHex(spki);
      receiverPrivateKey.value = S.toHex(pkcs8);
      copyReceiverPubBtn.disabled = false;
      copyReceiverPrivBtn.disabled = false;

      receiverPrivateKeyObj = keyPair.privateKey;

      S.showToast("RSA key pair generated");
    }, { overlay: "loadingOverlay", controls: [generateKeyPairBtn, clearReceiverBtn] });
  }
  generateKeyPairBtn.addEventListener("click", generateReceiverKeys);

  copyReceiverPubBtn.addEventListener("click", function () {
    S.copyToClipboard(receiverPublicKey.value, "Public key copied");
  });
  copyReceiverPrivBtn.addEventListener("click", function () {
    S.copyToClipboard(receiverPrivateKey.value, "Private key copied – keep it secret!");
  });

  async function ensureReceiverPrivateKey() {
    if (receiverPrivateKeyObj) return receiverPrivateKeyObj;
    var hex = receiverPrivateKey.value.trim();
    if (!hex) { throw new Error("No private key available."); }
    var pkcs8 = S.fromHex(hex);
    receiverPrivateKeyObj = await window.crypto.subtle.importKey(
      "pkcs8",
      pkcs8,
      { name: "RSA-OAEP", hash: "SHA-256" },
      false,
      ["decrypt"]
    );
    return receiverPrivateKeyObj;
  }

  async function decryptForReceiver() {
    var hex = ciphertextInput.value.trim();
    if (!hex) { S.showToast("Paste ciphertext first"); return; }

    decryptedOutput.value = "";
    copyDecryptedBtn.disabled = true;

    await S.runWithBusy(async function () {
      var priv = await ensureReceiverPrivateKey();
      var ct = S.fromHex(hex);
      var pt = await window.crypto.subtle.decrypt({ name: "RSA-OAEP" }, priv, ct);
      var decoded = S.decodeText(pt);
      decryptedOutput.value = decoded;
      copyDecryptedBtn.disabled = !decoded;
      S.showToast("Decrypted successfully");
    }, { overlay: "loadingOverlay", controls: [decryptBtn] });
  }
  decryptBtn.addEventListener("click", function () {
    decryptForReceiver().catch(function (err) { S.showModal("Decryption failed", String(err)); });
  });
  copyDecryptedBtn.addEventListener("click", function () {
    S.copyToClipboard(decryptedOutput.value, "Decrypted secret copied");
  });

  function clearSenderFields() {
    senderReceiverPubInput.value = "";
    senderPassword.value = "";
    encryptedOutput.value = "";
    copyEncryptedOutputBtn.disabled = true;
    senderWorkArea.classList.add("is-hidden");
    senderPublicKeyObj = null;
  }
  clearSenderBtn.addEventListener("click", clearSenderFields);

  async function useReceiverPublicKey() {
    var hex = senderReceiverPubInput.value.trim();
    if (!hex) { S.showToast("Paste a public key first"); return; }
    try {
      var spki = S.fromHex(hex);
      var key = await window.crypto.subtle.importKey(
        "spki",
        spki,
        { name: "RSA-OAEP", hash: "SHA-256" },
        false,
        ["encrypt"]
      );
      senderPublicKeyObj = key;
      senderWorkArea.classList.remove("is-hidden");
      S.showToast("Public key imported");
    } catch (err) {
      S.showModal("Invalid public key", "Could not import the provided public key: " + err);
    }
  }
  useReceiverPubBtn.addEventListener("click", function () { useReceiverPublicKey(); });

  generateSenderPasswordBtn.addEventListener("click", function () {
    var rnd = S.generateRandomString({ length: 32, label: "PASSWORD" });
    senderPassword.value = rnd.value;
    if (rnd.ok) { S.showToast("Random secret generated"); }
    else { S.showModal("Randomness warning", rnd.message); }
  });

  async function encryptForSender() {
    if (!senderPublicKeyObj) { S.showToast("Import receiver’s public key first"); return; }
    var secret = senderPassword.value.trim();
    if (!secret) { S.showToast("Type or generate a password first"); return; }

    var encoded = S.encodeText(secret);
    var enc = await window.crypto.subtle.encrypt({ name: "RSA-OAEP" }, senderPublicKeyObj, encoded);
    encryptedOutput.value = S.toHex(enc);
    copyEncryptedOutputBtn.disabled = false;
    S.showToast("Encrypted successfully");
  }
  encryptBtn.addEventListener("click", function () {
    encryptForSender().catch(function (err) { S.showModal("Encryption failed", String(err)); });
  });
  copyEncryptedOutputBtn.addEventListener("click", function () {
    S.copyToClipboard(encryptedOutput.value, "Encrypted payload copied");
  });

  // start on receiver
  switchToReceiver();
})();
