(function (global) {
    "use strict";
    var d = document;

    function getToastEl() { return d.getElementById("toast"); }
    function getModalEls() {
        var backdrop = d.getElementById("modalBackdrop");
        return {
            backdrop: backdrop,
            modal: backdrop ? backdrop.querySelector(".modal") : null,
            title: d.getElementById("modalTitle"),
            body: d.getElementById("modalBody")
        };
    }
    function getOverlay(elOrId) {
        if (elOrId && elOrId instanceof HTMLElement) return elOrId;
        if (typeof elOrId === "string") return d.getElementById(elOrId);
        return d.getElementById("loadingOverlay");
    }
    function toElementArray(list) {
        if (!list) return [];
        return list.map(function (item) {
            if (!item) return null;
            if (item instanceof HTMLElement) return item;
            if (typeof item === "string") return d.getElementById(item);
            return null;
        }).filter(Boolean);
    }

    function showToast(msg) {
        var toast = getToastEl();
        if (!toast) return;
        toast.textContent = msg;
        toast.classList.add("show");
        setTimeout(function () { toast.classList.remove("show"); }, 2600);
    }
    function showModal(title, message, variant) {
        var m = getModalEls();
        if (!m.backdrop) return;
        if (m.modal) {
            m.modal.classList.remove("success");
            if (variant === "success") m.modal.classList.add("success");
        }
        if (m.title) m.title.textContent = title;
        if (m.body) m.body.textContent = message;
        m.backdrop.style.display = "flex";
    }
    function hideModal() {
        var m = getModalEls();
        if (m.backdrop) m.backdrop.style.display = "none";
    }

    function restartSpinner(overlayEl) {
        var overlay = getOverlay(overlayEl);
        if (!overlay) return;
        var spinner = overlay.querySelector(".spinner");
        if (!spinner) return;
        var fresh = spinner.cloneNode(true);
        spinner.replaceWith(fresh);
    }
    function setControlsDisabled(controls, disabled) {
        controls.forEach(function (el) { if (el) el.disabled = disabled; });
    }
    async function runWithBusy(fn, options) {
        options = options || {};
        var overlay = getOverlay(options.overlay);
        var controls = toElementArray(options.controls || []);
        setControlsDisabled(controls, true);
        if (overlay) {
            overlay.style.display = "flex";
            restartSpinner(overlay);
            await new Promise(function (r) { requestAnimationFrame(r); });
            await new Promise(function (r) { requestAnimationFrame(r); });
        }
        try {
            await fn();
        } finally {
            setControlsDisabled(controls, false);
            if (overlay) overlay.style.display = "none";
        }
    }

    function fallbackCopy(text, onSuccess, onError) {
        try {
            var ta = d.createElement("textarea");
            ta.value = text;
            ta.style.position = "fixed";
            ta.style.top = "-9999px";
            d.body.appendChild(ta);
            ta.focus();
            ta.select();
            var ok = d.execCommand("copy");
            d.body.removeChild(ta);
            ok ? (onSuccess && onSuccess()) : (onError && onError());
        } catch (err) { if (onError) onError(err); }
    }
    function copyToClipboard(text, successMsg) {
        if (!text) { showToast("Nothing to copy"); return; }
        if (navigator.clipboard && window.isSecureContext) {
            navigator.clipboard.writeText(text).then(function () {
                showToast(successMsg || "Copied");
            }).catch(function () {
                fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                                   function () { showToast("Copy failed"); });
            });
        } else {
            fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                               function () { showToast("Copy failed"); });
        }
    }

    function hasWebCrypto() {
        return typeof window !== "undefined" && window.crypto && typeof window.crypto.getRandomValues === "function";
    }
    function withMathRandomTrap(fn) {
        var original = Math.random;
        var mathRandomUsed = false;
        Math.random = function () { mathRandomUsed = true; return original.call(Math); };
        var result, error;
        try { result = fn(); } catch (err) { error = err; } finally { Math.random = original; }
        return { result: result, error: error, mathRandomUsed: mathRandomUsed };
    }
    function evaluateRandomResult(label, info) {
        var hasSecure = !!info.hasSecure;
        var mathRandomUsed = !!info.mathRandomUsed;
        if (!hasSecure) {
            return { ok: false, variant: null, message: "❌ THIS " + label + " IS NOT SECURE ❌ This browser did NOT provide crypto.getRandomValues(). Use a modern browser, offline." };
        }
        if (mathRandomUsed) {
            return { ok: false, variant: null, message: "❌ THIS " + label + " IS NOT SECURE ❌ The library fell back to Math.random(). Use another browser/build." };
        }
        return { ok: true, variant: "success", message: "✅ Secure " + label.toLowerCase() + " generated successfully." };
    }

    function tableToCsv(tbody, headers) {
        var rows = [];
        rows.push(headers.join(","));
        var trs = tbody.querySelectorAll("tr");
        for (var i = 0; i < trs.length; i++) {
            var tds = trs[i].querySelectorAll("td");
            var cols = [];
            for (var j = 0; j < tds.length; j++) {
                var raw = tds[j].textContent.trim();
                cols.push('"' + raw.replace(/"/g, '""') + '"');
            }
            rows.push(cols.join(","));
        }
        return rows.join("\n");
    }
    function normalizeMnemonic(m) { return (m || "").trim().toLowerCase().replace(/\s+/g, " "); }

    global.AppShared = {
        showToast: showToast,
        showModal: showModal,
        hideModal: hideModal,
        runWithBusy: runWithBusy,
        copyToClipboard: copyToClipboard,
        hasWebCrypto: hasWebCrypto,
        withMathRandomTrap: withMathRandomTrap,
        evaluateRandomResult: evaluateRandomResult,
        tableToCsv: tableToCsv,
        normalizeMnemonic: normalizeMnemonic
    };

    d.addEventListener("click", function (e) {
        var m = getModalEls();
        if (e.target && e.target.id === "modalCloseBtn") { hideModal(); return; }
        if (m.backdrop && e.target === m.backdrop) { hideModal(); }
    });
    d.addEventListener("keydown", function (e) {
        var m = getModalEls();
        if (e.key === "Escape" && m.backdrop && m.backdrop.style.display === "flex") { hideModal(); }
    });
})(typeof window !== "undefined" ? window : this);

(function () {
    "use strict";
    var S = window.AppShared;
    var d = document;

    var modeBtns = document.querySelectorAll(".mode-btn");
    var panels = document.querySelectorAll(".mode-panel");

    function switchMode(mode) {
        modeBtns.forEach(function (b) { b.classList.toggle("is-active", b.getAttribute("data-mode") === mode); });
        panels.forEach(function (p) { p.classList.toggle("is-visible", p.id === "mode-" + mode); });
        clearAll();
    }
    modeBtns.forEach(function (b) {
        b.addEventListener("click", function () { switchMode(b.getAttribute("data-mode")); });
    });

    var mnemonicTA = document.getElementById("mnemonic");
    var passphraseIn = document.getElementById("passphrase");
    var generateMnemonicBtn = document.getElementById("generateMnemonicBtn");
    var copyMnemonicBtn = document.getElementById("copyMnemonic");
    var startIndexIn = document.getElementById("startIndex");
    var countIn = document.getElementById("count");
    var mnemonicLengthSel = document.getElementById("mnemonicLength");

    var xpub44In = document.getElementById("xpub44");
    var zpub84In = document.getElementById("zpub84");
    var ethXpubIn = document.getElementById("ethXpub");
    var startIndex2 = document.getElementById("startIndex2");
    var count2 = document.getElementById("count2");

    var xprv44In = document.getElementById("xprv44");
    var zprv84In = document.getElementById("zprv84");
    var ethXprvIn = document.getElementById("ethXprv");
    var startIndex3 = document.getElementById("startIndex3");
    var count3 = document.getElementById("count3");

    var deriveFromMnemonicBtn = document.getElementById("deriveFromMnemonicBtn");
    var deriveFromPubBtn = document.getElementById("deriveFromPubBtn");
    var deriveFromPrvBtn = document.getElementById("deriveFromPrvBtn");
    var clearBtn1 = document.getElementById("clearBtn1");
    var clearBtn2 = document.getElementById("clearBtn2");
    var clearBtn3 = document.getElementById("clearBtn3");

    var btcLegacyTbody = document.querySelector("#btcLegacyTable tbody");
    var btcLegacyChangeTbody = document.querySelector("#btcLegacyChangeTable tbody");
    var btcSegwitTbody = document.querySelector("#btcSegwitTable tbody");
    var btcSegwitChangeTbody = document.querySelector("#btcSegwitChangeTable tbody");
    var ethTbody = document.querySelector("#ethTable tbody");
    var btcLegacyCount = document.getElementById("btcLegacyCount");
    var btcSegwitCount = document.getElementById("btcSegwitCount");
    var ethCount = document.getElementById("ethCount");

    var btcLegacyXprvOut = document.getElementById("btcLegacyXprv");
    var btcLegacyXpubOut = document.getElementById("btcLegacyXpub");
    var btcSegwitZprvOut = document.getElementById("btcSegwitZprv");
    var btcSegwitZpubOut = document.getElementById("btcSegwitZpub");
    var ethXprvOut = document.getElementById("ethXprvOut");
    var ethXpubOut = document.getElementById("ethXpubOut");

    var exportBtns = document.querySelectorAll("button[data-export]");
    var copyBtns = document.querySelectorAll("button[data-copy]");

    var BUSY_CONTROLS = [
        mnemonicTA, passphraseIn, generateMnemonicBtn, copyMnemonicBtn, startIndexIn, countIn, mnemonicLengthSel,
        xpub44In, zpub84In, ethXpubIn, startIndex2, count2,
        xprv44In, zprv84In, ethXprvIn, startIndex3, count3,
        deriveFromMnemonicBtn, deriveFromPubBtn, deriveFromPrvBtn, clearBtn1, clearBtn2, clearBtn3
    ];

    function clearTables() {
        btcLegacyTbody.innerHTML = "";
        btcLegacyChangeTbody.innerHTML = "";
        btcSegwitTbody.innerHTML = "";
        btcSegwitChangeTbody.innerHTML = "";
        ethTbody.innerHTML = "";
        btcLegacyCount.textContent = "0";
        btcSegwitCount.textContent = "0";
        ethCount.textContent = "0";
    }
    function clearMasterOutputs() {
        if (btcLegacyXprvOut) btcLegacyXprvOut.value = "";
        if (btcLegacyXpubOut) btcLegacyXpubOut.value = "";
        if (btcSegwitZprvOut) btcSegwitZprvOut.value = "";
        if (btcSegwitZpubOut) btcSegwitZpubOut.value = "";
        if (ethXprvOut) ethXprvOut.value = "";
        if (ethXpubOut) ethXpubOut.value = "";
    }
    function clearAll() {
        mnemonicTA.value = "";
        passphraseIn.value = "";
        xpub44In.value = "";
        zpub84In.value = "";
        ethXpubIn.value = "";
        xprv44In.value = "";
        zprv84In.value = "";
        ethXprvIn.value = "";
        clearMasterOutputs();
        clearTables();
    }

    function addTableRow(tbody, index, address, priv, prefix) {
        var tr = d.createElement("tr");
        var tdIdx = d.createElement("td"); tdIdx.textContent = index; tdIdx.style.fontWeight = "600";
        var tdAddr = d.createElement("td"); tdAddr.innerHTML = "<code>" + address + "</code>";
        var tdPriv = d.createElement("td");
        var prefixText = prefix ? prefix + ":" : "";
        tdPriv.innerHTML = priv ? ("<code>" + prefixText + priv + "</code>") : "";
        tr.appendChild(tdIdx); tr.appendChild(tdAddr); tr.appendChild(tdPriv);
        tbody.appendChild(tr);
    }

    var ALPHABET = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz";
    var ALPHABET_MAP = {}; for (var i = 0; i < ALPHABET.length; i++) ALPHABET_MAP[ALPHABET.charAt(i)] = i;
    function sha256(buf) { return bitcoinjs.crypto.sha256(buf); }
    function doubleSha256Bytes(u8) { return sha256(sha256(u8)); }
    function base58decode(str) {
        var bytes = [0];
        for (var i = 0; i < str.length; i++) {
            var c = str[i];
            var val = ALPHABET_MAP[c];
            if (val === undefined) throw new Error("Invalid base58 character");
            for (var j = 0; j < bytes.length; j++) val += bytes[j] * 58, bytes[j] = val & 0xff, val >>= 8;
            while (val) bytes.push(val & 0xff), val >>= 8;
        }
        for (var k = 0; k < str.length && str[k] === "1"; k++) bytes.push(0);
        return Uint8Array.from(bytes.reverse());
    }
    function base58encode(buf) {
        var digits = [0];
        for (var i = 0; i < buf.length; i++) {
            var carry = buf[i];
            for (var j = 0; j < digits.length; j++) carry += digits[j] << 8, digits[j] = carry % 58, carry = (carry / 58) | 0;
            while (carry) digits.push(carry % 58), carry = (carry / 58) | 0;
        }
        for (var k = 0; k < buf.length && buf[k] === 0; k++) digits.push(0);
        return digits.reverse().map(function (d) { return ALPHABET[d]; }).join("");
    }
    function b58checkDecodeToPayload(b58) {
        var raw = base58decode(b58);
        if (raw.length < 4) throw new Error("Invalid base58check length");
        var data = raw.slice(0, raw.length - 4);
        var chk = raw.slice(raw.length - 4);
        var exp = doubleSha256Bytes(data).slice(0, 4);
        for (var i = 0; i < 4; i++) if (chk[i] !== exp[i]) throw new Error("Invalid base58check checksum");
        return data;
    }
    function b58checkEncodeFromPayload(payload) {
        var checksum = doubleSha256Bytes(payload).slice(0, 4);
        var raw = new Uint8Array(payload.length + 4);
        raw.set(payload, 0); raw.set(checksum, payload.length);
        return base58encode(raw);
    }
    var VERSIONS = {
        xpub: 0x0488B21E, xprv: 0x0488ADE4,
        ypub: 0x049D7CB2, yprv: 0x049D7878,
        zpub: 0x04B24746, zprv: 0x04B2430C
    };
    function swapVersion(b58, toPrefix) {
        var target = VERSIONS[toPrefix];
        if (typeof target !== "number") throw new Error("Unknown target prefix");
        var payload = b58checkDecodeToPayload(b58);
        if (payload.length < 4) throw new Error("Bad payload");
        var newPayload = new Uint8Array(payload);
        newPayload[0] = (target >>> 24) & 0xff;
        newPayload[1] = (target >>> 16) & 0xff;
        newPayload[2] = (target >>> 8) & 0xff;
        newPayload[3] = (target >>> 0) & 0xff;
        return b58checkEncodeFromPayload(newPayload);
    }
    function toXpubCompatible(b58) {
        var p = (b58 || "").slice(0, 4);
        if (p === "xpub" || p === "xprv") return b58;
        if (p === "ypub") return swapVersion(b58, "xpub");
        if (p === "yprv") return swapVersion(b58, "xprv");
        if (p === "zpub") return swapVersion(b58, "xpub");
        if (p === "zprv") return swapVersion(b58, "xprv");
        throw new Error("Unsupported key prefix");
    }
    function toSlip132(prefix, xkeyB58) {
        var p = prefix;
        if (p === "xpub" || p === "xprv") return xkeyB58;
        return swapVersion(xkeyB58, p);
    }

    function addrsFromBitcoinNode(accountNode, type, start, count, change) {
        var net = bitcoinjs.networks.bitcoin;
        var chain = change ? 1 : 0;
        var res = [];
        for (var i = start; i < start + count; i++) {
            var child = accountNode.derive(chain).derive(i);
            var pub = child.publicKey;
            var addr;
            if (type === "p2pkh") {
                addr = bitcoinjs.payments.p2pkh({ pubkey: pub, network: net }).address;
            } else {
                addr = bitcoinjs.payments.p2wpkh({ pubkey: pub, network: net }).address;
            }
            res.push({ index: i, address: addr, wif: child.privateKey ? bitcoinjs.ECPair.fromPrivateKey(child.privateKey, { network: net }).toWIF() : "" });
        }
        return res;
    }
    function addrsFromEthNode(accountNode, start, count) {
        var out = [];
        for (var i = start; i < start + count; i++) {
            var node = accountNode.derivePath("0/" + i);
            out.push({ index: i, address: node.address, priv: node.privateKey || "" });
        }
        return out;
    }

    function ensureLibs() {
        if (typeof bip39 === "undefined") throw new Error("bip39 library not loaded.");
        if (typeof bitcoinjs === "undefined" || !bitcoinjs.bip32 || !bitcoinjs.payments) throw new Error("bitcoinjs library not loaded.");
        if (typeof ethers === "undefined" || !ethers.utils || !ethers.utils.HDNode) throw new Error("ethers library not loaded.");
    }
    function updateCounts() {
        btcLegacyCount.textContent = String(btcLegacyTbody.querySelectorAll("tr").length);
        btcSegwitCount.textContent = String(btcSegwitTbody.querySelectorAll("tr").length);
        ethCount.textContent = String(ethTbody.querySelectorAll("tr").length);
    }

    async function deriveFromMnemonic() {
        ensureLibs();
        var mnemonic = S.normalizeMnemonic(mnemonicTA.value);
        var passphrase = passphraseIn.value || "";
        var startIndex = Number(startIndexIn.value || 0);
        var count = Number(countIn.value || 0); if (!count || count <= 0) count = 50; if (count > 1000) count = 1000;

        if (!mnemonic) throw new Error("Mnemonic is empty.");
        var words = mnemonic.split(" ");
        if ([12, 15, 18, 21, 24].indexOf(words.length) === -1) throw new Error("Mnemonic must have 12, 15, 18, 21, or 24 words.");
        if (!bip39.validateMnemonic(mnemonic)) throw new Error("This is not a valid BIP39 mnemonic. Checksum failed.");

        var seed = await bip39.mnemonicToSeed(mnemonic, passphrase);
        clearTables(); clearMasterOutputs();

        var net = bitcoinjs.networks.bitcoin;
        var rootBtc = bitcoinjs.bip32.fromSeed(seed, net);

        var legacyAccount = rootBtc.derivePath("m/44'/0'/0'");
        var segwitAccount = rootBtc.derivePath("m/84'/0'/0'");

        var legacyXprv = legacyAccount.toBase58();
        var legacyXpub = legacyAccount.neutered().toBase58();
        var segwitZprv = toSlip132("zprv", segwitAccount.toBase58());
        var segwitZpub = toSlip132("zpub", segwitAccount.neutered().toBase58());

        btcLegacyXprvOut.value = legacyXprv;
        btcLegacyXpubOut.value = legacyXpub;
        btcSegwitZprvOut.value = segwitZprv;
        btcSegwitZpubOut.value = segwitZpub;

        var ethRoot = ethers.utils.HDNode.fromMnemonic(mnemonic, passphrase);
        var ethAccount = ethRoot.derivePath("m/44'/60'/0'");
        ethXprvOut.value = ethAccount.extendedKey;
        ethXpubOut.value = ethAccount.neuter().extendedKey;

        var legacyRows = addrsFromBitcoinNode(legacyAccount, "p2pkh", startIndex, count, false);
        var legacyChangeRows = addrsFromBitcoinNode(legacyAccount, "p2pkh", startIndex, count, true);
        var segwitRows = addrsFromBitcoinNode(segwitAccount, "p2wpkh", startIndex, count, false);
        var segwitChangeRows = addrsFromBitcoinNode(segwitAccount, "p2wpkh", startIndex, count, true);
        legacyRows.forEach(function (r) { addTableRow(btcLegacyTbody, r.index, r.address, r.wif, "p2pkh"); });
        legacyChangeRows.forEach(function (r) { addTableRow(btcLegacyChangeTbody, r.index, r.address, r.wif, "p2pkh"); });
        segwitRows.forEach(function (r) { addTableRow(btcSegwitTbody, r.index, r.address, r.wif, "p2wpkh"); });
        segwitChangeRows.forEach(function (r) { addTableRow(btcSegwitChangeTbody, r.index, r.address, r.wif, "p2wpkh"); });

        var ethRows = addrsFromEthNode(ethAccount, startIndex, count);
        ethRows.forEach(function (r) { addTableRow(ethTbody, r.index, r.address, r.priv); });

        updateCounts();
    }

    function parseBtcAccountFromB58(b58) {
        var net = bitcoinjs.networks.bitcoin;
        var xKey = toXpubCompatible(b58);
        return bitcoinjs.bip32.fromBase58(xKey, net);
    }
    function deriveFromPublic() {
        ensureLibs();
        clearTables(); clearMasterOutputs();

        var start = Number(startIndex2.value || 0);
        var count = Number(count2.value || 0); if (!count || count <= 0) count = 50; if (count > 1000) count = 1000;

        if (xpub44In.value.trim()) {
            var acc44 = parseBtcAccountFromB58(xpub44In.value.trim()).neutered();
            addrsFromBitcoinNode(acc44, "p2pkh", start, count, false)
                .forEach(function (r) { addTableRow(btcLegacyTbody, r.index, r.address, ""); });
            addrsFromBitcoinNode(acc44, "p2pkh", start, count, true)
                .forEach(function (r) { addTableRow(btcLegacyChangeTbody, r.index, r.address, ""); });
            btcLegacyXpubOut.value = toSlip132("xpub", acc44.toBase58());
        }
        if (zpub84In.value.trim()) {
            var acc84 = parseBtcAccountFromB58(zpub84In.value.trim()).neutered();
            addrsFromBitcoinNode(acc84, "p2wpkh", start, count, false)
                .forEach(function (r) { addTableRow(btcSegwitTbody, r.index, r.address, ""); });
            addrsFromBitcoinNode(acc84, "p2wpkh", start, count, true)
                .forEach(function (r) { addTableRow(btcSegwitChangeTbody, r.index, r.address, ""); });
            btcSegwitZpubOut.value = toSlip132("zpub", acc84.toBase58());
        }
        if (ethXpubIn.value.trim()) {
            var ethAccPub = ethers.utils.HDNode.fromExtendedKey(ethXpubIn.value.trim()).neuter();
            ethXpubOut.value = ethAccPub.extendedKey;
            addrsFromEthNode(ethAccPub, start, count)
                .forEach(function (r) { addTableRow(ethTbody, r.index, r.address, ""); });
        }
        updateCounts();
        if (!btcLegacyTbody.children.length && !btcSegwitTbody.children.length && !ethTbody.children.length) {
            throw new Error("Provide at least one master public key.");
        }
    }
    function deriveFromPrivate() {
        ensureLibs();
        clearTables(); clearMasterOutputs();

        var start = Number(startIndex3.value || 0);
        var count = Number(count3.value || 0); if (!count || count <= 0) count = 50; if (count > 1000) count = 1000;

        if (xprv44In.value.trim()) {
            var acc44 = parseBtcAccountFromB58(xprv44In.value.trim());
            addrsFromBitcoinNode(acc44, "p2pkh", start, count, false)
                .forEach(function (r) { addTableRow(btcLegacyTbody, r.index, r.address, r.wif, "p2pkh"); });
            addrsFromBitcoinNode(acc44, "p2pkh", start, count, true)
                .forEach(function (r) { addTableRow(btcLegacyChangeTbody, r.index, r.address, r.wif, "p2pkh"); });
            btcLegacyXprvOut.value = toSlip132("xprv", acc44.toBase58());
            btcLegacyXpubOut.value = toSlip132("xpub", acc44.neutered().toBase58());
        }
        if (zprv84In.value.trim()) {
            var acc84 = parseBtcAccountFromB58(zprv84In.value.trim());
            addrsFromBitcoinNode(acc84, "p2wpkh", start, count, false)
                .forEach(function (r) { addTableRow(btcSegwitTbody, r.index, r.address, r.wif, "p2wpkh"); });
            addrsFromBitcoinNode(acc84, "p2wpkh", start, count, true)
                .forEach(function (r) { addTableRow(btcSegwitChangeTbody, r.index, r.address, r.wif, "p2wpkh"); });
            btcSegwitZprvOut.value = toSlip132("zprv", acc84.toBase58());
            btcSegwitZpubOut.value = toSlip132("zpub", acc84.neutered().toBase58());
        }
        if (ethXprvIn.value.trim()) {
            var ethAcc = ethers.utils.HDNode.fromExtendedKey(ethXprvIn.value.trim());
            ethXprvOut.value = ethAcc.extendedKey;
            ethXpubOut.value = ethAcc.neuter().extendedKey;
            addrsFromEthNode(ethAcc, start, count)
                .forEach(function (r) { addTableRow(ethTbody, r.index, r.address, r.priv); });
        }
        updateCounts();
        if (!btcLegacyTbody.children.length && !btcSegwitTbody.children.length && !ethTbody.children.length) {
            throw new Error("Provide at least one master private key.");
        }
    }

    function generateRandomMnemonic() {
        if (typeof bip39 === "undefined" || typeof bip39.generateMnemonic !== "function") {
            S.showToast("bip39.generateMnemonic() not available.");
            return;
        }
        var selectedLen = Number(mnemonicLengthSel && mnemonicLengthSel.value ? mnemonicLengthSel.value : 24) || 24;
        var wordToEntropy = { 12: 128, 15: 160, 18: 192, 21: 224, 24: 256 };
        var entropy = wordToEntropy[selectedLen] || 256;
        var trapped = S.withMathRandomTrap(function () { return bip39.generateMnemonic(entropy); });
        if (trapped.error) { S.showModal("Mnemonic generation failed", trapped.error.message || String(trapped.error)); return; }
        mnemonicTA.value = trapped.result;
        passphraseIn.value = "";
        var evalRes = S.evaluateRandomResult("MNEMONIC", { hasSecure: S.hasWebCrypto(), mathRandomUsed: trapped.mathRandomUsed });
        S.showToast("Random " + selectedLen + "-word mnemonic generated.");
        if (!evalRes.ok) S.showModal("Mnemonic generated", evalRes.message);
    }

    deriveFromMnemonicBtn.addEventListener("click", function () {
        S.runWithBusy(async function () {
            try { await deriveFromMnemonic(); }
            catch (err) { S.showModal("Derivation failed", err.message || String(err)); }
        }, { controls: BUSY_CONTROLS, overlay: "loadingOverlay" });
    });
    deriveFromPubBtn.addEventListener("click", function () {
        S.runWithBusy(async function () {
            try { deriveFromPublic(); }
            catch (err) { S.showModal("Derivation failed", err.message || String(err)); }
        }, { controls: BUSY_CONTROLS, overlay: "loadingOverlay" });
    });
    deriveFromPrvBtn.addEventListener("click", function () {
        S.runWithBusy(async function () {
            try { deriveFromPrivate(); }
            catch (err) { S.showModal("Derivation failed", err.message || String(err)); }
        }, { controls: BUSY_CONTROLS, overlay: "loadingOverlay" });
    });

    clearBtn1.addEventListener("click", clearAll);
    clearBtn2.addEventListener("click", clearAll);
    clearBtn3.addEventListener("click", clearAll);

    generateMnemonicBtn.addEventListener("click", function () { generateRandomMnemonic(); });
    copyMnemonicBtn.addEventListener("click", function () { S.copyToClipboard(mnemonicTA.value, "Mnemonic copied"); });

    exportBtns.forEach(function (btn) {
        btn.addEventListener("click", function () {
            var kind = btn.getAttribute("data-export");
            var csv = "";
            if (kind === "btc-legacy") {
                csv = S.tableToCsv(btcLegacyTbody, ["index", "address", "private_key_wif"]);
            } else if (kind === "btc-legacy-change") {
                csv = S.tableToCsv(btcLegacyChangeTbody, ["index", "address", "private_key_wif"]);
            } else if (kind === "btc-segwit") {
                csv = S.tableToCsv(btcSegwitTbody, ["index", "address", "private_key_wif"]);
            } else if (kind === "btc-segwit-change") {
                csv = S.tableToCsv(btcSegwitChangeTbody, ["index", "address", "private_key_wif"]);
            } else if (kind === "eth") {
                csv = S.tableToCsv(ethTbody, ["index", "address", "private_key_hex"]);
            }
            if (!csv.trim()) { S.showToast("Nothing to export"); return; }
            S.copyToClipboard(csv, "CSV copied");
        });
    });
    copyBtns.forEach(function (btn) {
        btn.addEventListener("click", function () {
            var sel = btn.getAttribute("data-copy");
            var el = sel ? document.querySelector(sel) : null;
            if (!el) { S.showToast("Nothing to copy"); return; }
            S.copyToClipboard(el.value, "Copied");
        });
    });

    switchMode("mnemonic");
})();
