(function (global) {
  "use strict";
  var d = document;

  function $(id) { return d.getElementById(id); }

  function getToastEl() { return $("toast"); }
  function getModalEls() {
    var backdrop = $("modalBackdrop");
    return {
      backdrop: backdrop,
      modal: backdrop ? backdrop.querySelector(".modal") : null,
      title: $("modalTitle"),
      body: $("modalBody")
    };
  }
  function showToast(msg) {
    var toast = getToastEl();
    if (!toast) return;
    toast.textContent = msg;
    toast.classList.add("show");
    setTimeout(function () { toast.classList.remove("show"); }, 2600);
  }
  function showModal(title, message, variant) {
    var m = getModalEls();
    if (!m.backdrop) return;
    if (m.modal) {
      m.modal.classList.remove("success");
      if (variant === "success") m.modal.classList.add("success");
    }
    if (m.title) m.title.textContent = title;
    if (m.body) m.body.textContent = message;
    m.backdrop.style.display = "flex";
  }
  function hideModal() {
    var m = getModalEls();
    if (m.backdrop) m.backdrop.style.display = "none";
  }
  d.addEventListener("click", function (e) {
    if (e.target && e.target.id === "modalCloseBtn") { hideModal(); return; }
    var m = getModalEls();
    if (m.backdrop && e.target === m.backdrop) { hideModal(); }
  });
  d.addEventListener("keydown", function (e) {
    var m = getModalEls();
    if (e.key === "Escape" && m.backdrop && m.backdrop.style.display === "flex") hideModal();
  });

  function getOverlay() { return $("loadingOverlay"); }
  function restartSpinner() {
    var overlay = getOverlay();
    if (!overlay) return;
    var spinner = overlay.querySelector(".spinner");
    if (!spinner) return;
    var fresh = spinner.cloneNode(true);
    spinner.replaceWith(fresh);
  }
  async function runWithBusy(fn, controls) {
    var overlay = getOverlay();
    var list = (controls || []).map(function (x) { return typeof x === "string" ? $(x) : x; }).filter(Boolean);
    list.forEach(function (el) { el.disabled = true; });
    if (overlay) {
      overlay.style.display = "flex";
      restartSpinner();
      await new Promise(function (r) { requestAnimationFrame(r); });
      await new Promise(function (r) { requestAnimationFrame(r); });
    }
    try { await fn(); }
    finally {
      list.forEach(function (el) { el.disabled = false; });
      if (overlay) overlay.style.display = "none";
    }
  }

  function fallbackCopy(text, onSuccess, onError) {
    try {
      var ta = d.createElement("textarea");
      ta.value = text;
      ta.style.position = "fixed";
      ta.style.top = "-9999px";
      d.body.appendChild(ta);
      ta.focus(); ta.select();
      var ok = d.execCommand("copy");
      d.body.removeChild(ta);
      ok ? (onSuccess && onSuccess()) : (onError && onError());
    } catch (err) {
      if (onError) onError(err);
    }
  }
  function copyToClipboard(text, successMsg) {
    if (!text) { showToast("Nothing to copy"); return; }
    if (navigator.clipboard && window.isSecureContext) {
      navigator.clipboard.writeText(text).then(function () {
        showToast(successMsg || "Copied");
      }).catch(function () {
        fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                          function () { showToast("Copy failed"); });
      });
    } else {
      fallbackCopy(text, function () { showToast(successMsg || "Copied"); },
                        function () { showToast("Copy failed"); });
    }
  }

  var textInput = $("textInput");
  var qrCanvas = $("qrCanvas");
  var downloadPngBtn = $("downloadPngBtn");
  var clearTextBtn = $("clearTextBtn");
  var encodeInfo = $("encodeInfo");
  var uploadBtn = $("uploadBtn");
  var scanBtn = $("scanBtn");
  var imageInput = $("imageInput");

  function toUtf8Bytes(str) { return new TextEncoder().encode(str); }
  function maxBytesForLevel(level) {
    if (level === "L") return 2953;
    if (level === "M") return 2331;
    if (level === "Q") return 1663;
    return 1273;
  }
  function canEncode(byteLength, level) { return byteLength <= maxBytesForLevel(level || "Q"); }
  function pickEccForSize(byteLength) {
    var levels = ["Q", "M", "L"];
    for (var i = 0; i < levels.length; i++) if (canEncode(byteLength, levels[i])) return levels[i];
    return null;
  }

  function drawQuietCanvas() {
    var ctx = qrCanvas.getContext("2d", { willReadFrequently: true });
    ctx.clearRect(0, 0, qrCanvas.width, qrCanvas.height);
    ctx.fillStyle = "#fff";
    ctx.fillRect(0, 0, qrCanvas.width, qrCanvas.height);
  }

  function renderQr(str) {
    if (global.qrcode && typeof global.qrcode.stringToBytes !== "function") {
      try { global.qrcode.stringToBytes = function (s) { return Array.from(new TextEncoder().encode(s)); }; } catch (_) {}
    }
    var bytes = toUtf8Bytes(str);
    if (!str) { drawQuietCanvas(); encodeInfo.textContent = "Status: Waiting for input…"; return; }
    var ecc = pickEccForSize(bytes.length);
    if (!ecc) { drawQuietCanvas(); encodeInfo.classList.add("bad"); encodeInfo.textContent = "⚠️ Input too large to fit into a single QR code."; return; }
    encodeInfo.classList.remove("bad");
    var qr = global.qrcode(0, ecc);
    qr.addData(str, "Byte");
    try { qr.make(); }
    catch (err) { drawQuietCanvas(); encodeInfo.classList.add("bad"); encodeInfo.textContent = "Encoding failed: " + (err && err.message ? err.message : err); return; }

    var ctx = qrCanvas.getContext("2d", { willReadFrequently: true });
    var moduleCount = qr.getModuleCount();
    var scale = Math.floor(Math.min(qrCanvas.width, qrCanvas.height) / (moduleCount + 8));
    var size = scale * (moduleCount + 8);
    var offset = Math.floor((qrCanvas.width - size) / 2);

    ctx.fillStyle = "#fff";
    ctx.fillRect(0, 0, qrCanvas.width, qrCanvas.height);
    ctx.fillStyle = "#000";

    for (var r = 0; r < moduleCount; r++) {
      for (var c = 0; c < moduleCount; c++) {
        if (qr.isDark(r, c)) {
          var x = offset + scale * (c + 4);
          var y = offset + scale * (r + 4);
          ctx.fillRect(x, y, scale, scale);
        }
      }
    }
    encodeInfo.textContent = "Encoded " + bytes.length + " bytes using ECC-" + ecc + ". Module size: " + moduleCount + "×" + moduleCount + ".";
  }

  var debounceTimer = 0;
  function debounceRender() {
    window.clearTimeout(debounceTimer);
    debounceTimer = window.setTimeout(function () { renderQr(textInput.value); }, 120);
  }

  function downloadCanvasPng(canvas, filename) {
    canvas.toBlob(function (blob) {
      if (!blob) { showToast("Download failed"); return; }
      var a = d.createElement("a");
      a.href = URL.createObjectURL(blob);
      a.download = filename || "qr.png";
      d.body.appendChild(a);
      a.click();
      setTimeout(function () { URL.revokeObjectURL(a.href); d.body.removeChild(a); }, 450);
    }, "image/png");
  }

  function readImageFile(file) {
    return new Promise(function (resolve, reject) {
      var reader = new FileReader();
      reader.onload = function (e) {
        var img = new Image();
        img.onload = function () { resolve(img); };
        img.onerror = reject;
        img.src = e.target.result;
      };
      reader.onerror = reject;
      reader.readAsDataURL(file);
    });
  }

  function imageToImageData(img, maxSide) {
    var side = Math.min(maxSide || 1400, Math.max(img.width, img.height));
    var scale = side / Math.max(img.width, img.height);
    var w = Math.round(img.width * scale);
    var h = Math.round(img.height * scale);
    var cv = d.createElement("canvas");
    cv.width = w; cv.height = h;
    var ctx = cv.getContext("2d", { willReadFrequently: true });
    ctx.drawImage(img, 0, 0, w, h);
    try { return ctx.getImageData(0, 0, w, h); } catch (e) { return null; }
  }

  async function handleDecodeFile(file) {
    if (!file) return;
    await runWithBusy(async function () {
      var img = await readImageFile(file);
      var data = imageToImageData(img, 1400);
      if (!data) { showModal("Decode Error", "Unable to read image data."); return; }
      var result = global.jsQR(data.data, data.width, data.height, { inversionAttempts: "attemptBoth" });
      if (result && result.data) {
        textInput.value = result.data;
        showToast("QR decoded");
        renderQr(textInput.value);
      } else {
        showModal("Decode Error", "No QR detected in the provided image.");
      }
    }, [uploadBtn, imageInput]);
  }

  // ---- Live camera scan (Scan QR) ----
  function isSecureForCamera() {
    if (window.isSecureContext) return true;
    var loc = window.location;
    return loc.hostname === "localhost" || loc.hostname === "127.0.0.1";
  }

  function createScanOverlay() {
    var wrap = d.createElement("div");
    wrap.id = "scanOverlay";
    wrap.style.position = "fixed";
    wrap.style.inset = "0";
    wrap.style.background = "rgba(0,0,0,0.85)";
    wrap.style.display = "flex";
    wrap.style.flexDirection = "column";
    wrap.style.alignItems = "center";
    wrap.style.justifyContent = "center";
    wrap.style.zIndex = "9999";

    var video = d.createElement("video");
    video.setAttribute("playsinline", "true");
    video.autoplay = true;
    video.muted = true;
    video.style.maxWidth = "92vw";
    video.style.maxHeight = "64vh";
    video.style.borderRadius = "16px";
    video.style.boxShadow = "0 0 0 2px rgba(255,255,255,0.2)";

    var note = d.createElement("div");
    note.textContent = "Point your camera at a QR code";
    note.style.color = "#fff";
    note.style.margin = "12px 0 8px";
    note.style.fontSize = "14px";

    var btnRow = d.createElement("div");
    btnRow.style.display = "flex";
    btnRow.style.gap = "8px";

    var cancel = d.createElement("button");
    cancel.type = "button";
    cancel.textContent = "Cancel";
    cancel.className = "btn secondary";
    cancel.style.minWidth = "96px";

    wrap.appendChild(video);
    wrap.appendChild(note);
    btnRow.appendChild(cancel);
    wrap.appendChild(btnRow);
    d.body.appendChild(wrap);
    return { el: wrap, video: video, cancelBtn: cancel };
  }

  function stopStream(stream) {
    if (!stream) return;
    stream.getTracks().forEach(function (t) { try { t.stop(); } catch (_) {} });
  }

  function chooseBestFacingConstraint() {
    return [
      { video: { facingMode: { exact: "environment" } } },
      { video: { facingMode: "environment" } },
      { video: { facingMode: { exact: "user" } } },
      { video: true }
    ];
  }

  async function scanFromCamera() {
    if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
      showModal("Camera Unavailable", "Your browser does not support camera access.");
      return;
    }
    if (!isSecureForCamera()) {
      showModal("Needs Secure Context", "Camera access requires HTTPS or localhost. Open this file via a local web server (e.g. http://localhost) to use scanning.");
      return;
    }

    var overlay = createScanOverlay();
    var video = overlay.video;
    var stream = null;
    var rafId = 0;
    var ended = false;

    function cleanup() {
      if (ended) return;
      ended = true;
      try { cancelAnimationFrame(rafId); } catch (_) {}
      stopStream(stream);
      if (overlay && overlay.el && overlay.el.parentNode) overlay.el.parentNode.removeChild(overlay.el);
    }

    overlay.cancelBtn.addEventListener("click", cleanup);
    d.addEventListener("keydown", function esc(e) {
      if (e.key === "Escape") { d.removeEventListener("keydown", esc); cleanup(); }
    });

    var constraintsList = chooseBestFacingConstraint();
    var lastErr = null;
    for (var i = 0; i < constraintsList.length; i++) {
      try {
        stream = await navigator.mediaDevices.getUserMedia(constraintsList[i]);
        break;
      } catch (err) { lastErr = err; }
    }
    if (!stream) {
      cleanup();
      var msg = (lastErr && lastErr.name === "NotAllowedError") ?
        "Camera permission denied." : "Unable to access camera.";
      showModal("Scan Error", msg);
      return;
    }

    video.srcObject = stream;
    await video.play().catch(function(){ /* ignored */ });

    var canvas = d.createElement("canvas");
    var ctx = canvas.getContext("2d", { willReadFrequently: true });

    var scanStart = Date.now();
    (function tick() {
      if (ended) return;
      if (video.readyState >= 2) {
        var vw = video.videoWidth || 640;
        var vh = video.videoHeight || 480;
        var maxSide = 900;
        var scale = Math.min(1, maxSide / Math.max(vw, vh));
        canvas.width = Math.floor(vw * scale);
        canvas.height = Math.floor(vh * scale);
        ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
        var imageData;
        try { imageData = ctx.getImageData(0, 0, canvas.width, canvas.height); } catch (_) {}
        if (imageData) {
          var result = global.jsQR(imageData.data, imageData.width, imageData.height, { inversionAttempts: "attemptBoth" });
          if (result && result.data) {
            cleanup();
            textInput.value = result.data;
            showToast("QR scanned");
            renderQr(textInput.value);
            return;
          }
        }
      }
      if (Date.now() - scanStart > 80000) { // safety timeout
        cleanup();
        showModal("No QR Found", "Timed out looking for a QR code. Try again with better lighting and focus.");
        return;
      }
      rafId = requestAnimationFrame(tick);
    })();
  }

  textInput.addEventListener("input", debounceRender);
  clearTextBtn.addEventListener("click", function () { textInput.value = ""; renderQr(""); });
  downloadPngBtn.addEventListener("click", function () { downloadCanvasPng(qrCanvas, "qrtool.png"); });

  uploadBtn.addEventListener("click", function () { imageInput.click(); });
  imageInput.addEventListener("change", function () {
    if (imageInput.files && imageInput.files.length) handleDecodeFile(imageInput.files[0]);
  });

  scanBtn.addEventListener("click", function () { scanFromCamera(); });

  renderQr("");

})(typeof window !== "undefined" ? window : this);
