"use strict";

const TOKEN_INFO = Object.freeze({
  ETH: Object.freeze({ type: "native", decimals: 18 }),
  USDT: Object.freeze({
    type: "erc20",
    address: "0xdAC17F958D2ee523a2206206994597C13D831ec7",
    decimals: 6
  }),
  USDC: Object.freeze({
    type: "erc20",
    address: "0xA0b86991c6218b36c1d19D4a2e9Eb0cE3606eB48",
    decimals: 6
  }),
  DAI: Object.freeze({
    type: "erc20",
    address: "0x6B175474E89094C44Da98b954EedeAC495271d0F",
    decimals: 18
  }),
  CBBTC: Object.freeze({
    type: "erc20",
    address: "0xcbB7C0000aB88B473b1f5aFd9ef808440eed33Bf",
    decimals: 8
  }),
  WBTC: Object.freeze({
    type: "erc20",
    address: "0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599",
    decimals: 8
  })
});

const TOKEN_BY_ADDRESS = (() => {
  const map = new Map();
  for (const [symbol, info] of Object.entries(TOKEN_INFO)) {
    if (info.type === "erc20") {
      map.set(info.address.toLowerCase(), { symbol, decimals: info.decimals });
    }
  }
  return map;
})();

const erc20Interface = new ethers.Interface(["function transfer(address to, uint256 amount)"]);

const $ = (id) => document.getElementById(id);
const setText = (id, text) => ($(id).textContent = text);
const setValue = (id, value) => ($(id).value = value);

let toastTimer = null;
function showToast(message) {
  const toast = $("toast");
  if (!toast) return;

  toast.textContent = message;
  toast.classList.add("show");

  if (toastTimer) clearTimeout(toastTimer);
  toastTimer = setTimeout(() => toast.classList.remove("show"), 2600);
}

function fallbackCopy(text, onSuccess, onError) {
  try {
    const ta = document.createElement("textarea");
    ta.value = text;
    ta.style.position = "fixed";
    ta.style.top = "-9999px";
    document.body.appendChild(ta);
    ta.focus();
    ta.select();

    const ok = document.execCommand("copy");
    document.body.removeChild(ta);

    ok ? onSuccess?.() : onError?.();
  } catch (err) {
    onError?.(err);
  }
}

function copyToClipboard(text, successMsg = "Copied") {
  if (!text) return showToast("Nothing to copy");

  if (navigator.clipboard && window.isSecureContext) {
    navigator.clipboard
      .writeText(text)
      .then(() => showToast(successMsg))
      .catch(() =>
        fallbackCopy(
          text,
          () => showToast(successMsg),
          () => showToast("Copy failed")
        )
      );
    return;
  }

  fallbackCopy(
    text,
    () => showToast(successMsg),
    () => showToast("Copy failed")
  );
}

function normalizeDecimal(input, label) {
  let s = (input ?? "").trim().replace(/\s+/g, "");
  if (!s) throw new Error(`Missing ${label}.`);

  if (/[eE]/.test(s)) throw new Error(`${label} must not use scientific notation.`);
  if (/[+-]/.test(s)) throw new Error(`${label} must be positive.`);
  if (/[_']/g.test(s)) throw new Error(`${label} contains invalid separators.`);

  const hasComma = s.includes(",");
  const hasDot = s.includes(".");
  if (hasComma && hasDot) {
    throw new Error(
      `${label} is ambiguous (contains both ',' and '.'). Use only one as decimal separator and no thousands separators.`
    );
  }
  if (hasComma) s = s.replace(",", ".");
  if (!/^\d+(\.\d+)?$/.test(s)) throw new Error(`Invalid ${label} format.`);

  return s;
}

function normalizeUint(input, label) {
  const s = (input ?? "").trim().replace(/\s+/g, "");
  if (!s) throw new Error(`Missing ${label}.`);
  if (!/^\d+$/.test(s)) throw new Error(`${label} must be a non-negative integer.`);
  return s;
}

function safeGetAddress(input, label) {
  const s = (input ?? "").trim();
  if (!s) throw new Error(`Missing ${label}.`);
  if (!ethers.isAddress(s)) throw new Error(`Invalid ${label}.`);
  return ethers.getAddress(s);
}

function safePrivateKey(input) {
  let s = (input ?? "").trim();
  if (!s) throw new Error("Missing private key.");
  if (!s.startsWith("0x")) s = `0x${s}`;
  if (!/^0x[0-9a-fA-F]{64}$/.test(s)) {
    throw new Error("Invalid private key format. Expected 32-byte hex.");
  }
  return s;
}

function formatUnitsSafe(value, decimals) {
  try {
    return ethers.formatUnits(value, decimals);
  } catch {
    return value.toString();
  }
}

function buildDecodedSummary(parsedTx) {
  const lines = [];

  if (parsedTx.from) lines.push(`From:          ${parsedTx.from}`);
  if (parsedTx.to) lines.push(`To:            ${parsedTx.to}`);

  if (typeof parsedTx.nonce === "number") lines.push(`Nonce:         ${parsedTx.nonce}`);
  if (parsedTx.gasLimit != null) lines.push(`Gas limit:     ${parsedTx.gasLimit.toString()}`);

  if (parsedTx.gasPrice != null) {
    lines.push(`Gas price:     ${ethers.formatUnits(parsedTx.gasPrice, "gwei")} gwei`);
  }

  if (parsedTx.value != null) {
    lines.push(`Value:         ${formatUnitsSafe(parsedTx.value, 18)} ETH`);
    lines.push(`Value (wei):   ${parsedTx.value.toString()}`);
  }

  const data = parsedTx.data ?? "0x";
  lines.push(`Data:          ${data}`);

  if (data !== "0x") {
    try {
      const decoded = erc20Interface.parseTransaction({ data });
      if (decoded?.name === "transfer") {
        const toArg = decoded.args[0];
        const amtArg = decoded.args[1];

        lines.push("", "Decoded call:");
        lines.push("  method:      transfer(address,uint256)");
        lines.push(`  to:          ${ethers.getAddress(toArg)}`);
        lines.push(`  amount:      ${amtArg.toString()} (raw)`);

        const tokenMeta = parsedTx.to ? TOKEN_BY_ADDRESS.get(parsedTx.to.toLowerCase()) : null;
        if (tokenMeta) {
          lines.push(`  token:       ${tokenMeta.symbol}`);
          lines.push(`  amount:      ${formatUnitsSafe(amtArg, tokenMeta.decimals)} ${tokenMeta.symbol}`);
        }
      }
    } catch {
      /* ignore non-ERC20 calldata */
    }
  }

  return lines.join("\n");
}

function clearAll() {
  setValue("fromPrivateKey", "");
  setValue("derivedFrom", "");
  setValue("toAddress", "");
  setValue("amount", "");
  setValue("nonce", "");
  setValue("gasLimit", "");
  setValue("gasPriceInGwei", "");
  $("coin").value = "ETH";

  setValue("signedTx", "");
  setText("txSummary", "—");
  setText("log", "—");
}

async function copySignedTx() {
  const signed = $("signedTx").value.trim();
  if (!signed) {
    setText("log", "Nothing to copy.");
    showToast("Nothing to copy");
    return;
  }

  copyToClipboard(signed, "Signed tx copied");
  setText("log", "Copied signed transaction hex to clipboard.");
}

async function createSignedTx() {
  const btn = $("btnSign");
  btn.disabled = true;

  setValue("signedTx", "");
  setText("txSummary", "—");
  setText("log", "—");

  try {
    const CHAIN_ID = 1n;

    const coin = $("coin").value;
    const tokenInfo = TOKEN_INFO[coin];
    if (!tokenInfo) throw new Error("Unsupported asset selected.");

    const wallet = new ethers.Wallet(safePrivateKey($("fromPrivateKey").value));
    const recipient = safeGetAddress($("toAddress").value, "recipient address");

    const amountStr = normalizeDecimal($("amount").value, "amount");
    const nonceStr = normalizeUint($("nonce").value, "nonce");
    const gasLimitStr = normalizeUint($("gasLimit").value, "gas limit");

    const gasPriceStr = normalizeDecimal($("gasPriceInGwei").value, "gas price (gwei)");
    const gasPrice = ethers.parseUnits(gasPriceStr, "gwei");
    if (gasPrice <= 0n) throw new Error("Gas price must be > 0.");

    const nonce = Number(nonceStr);
    if (!Number.isSafeInteger(nonce) || nonce < 0) throw new Error("Nonce is out of supported range.");

    const gasLimit = BigInt(gasLimitStr);
    if (gasLimit <= 0n) throw new Error("Gas limit must be > 0.");

    if (tokenInfo.type === "native" && gasLimit < 21000n) {
      throw new Error("ETH transfer gas limit is usually 21000 or higher.");
    }
    if (tokenInfo.type === "erc20" && gasLimit < 45000n) {
      throw new Error("Token transfers usually require > 45000 gas. Verify gas limit.");
    }

    const amount = ethers.parseUnits(amountStr, tokenInfo.decimals);
    if (amount <= 0n) throw new Error("Amount must be > 0.");

    let tx;
    if (tokenInfo.type === "native") {
      tx = {
        type: 0,
        to: recipient,
        value: amount,
        nonce,
        gasLimit,
        gasPrice,
        chainId: CHAIN_ID
      };
    } else {
      const tokenContract = safeGetAddress(tokenInfo.address, "token contract address");
      const data = erc20Interface.encodeFunctionData("transfer", [recipient, amount]);
      tx = {
        type: 0,
        to: tokenContract,
        value: 0n,
        data,
        nonce,
        gasLimit,
        gasPrice,
        chainId: CHAIN_ID
      };
    }

    const signedTx = await wallet.signTransaction(tx);
    setValue("signedTx", signedTx);

    const parsed = ethers.Transaction.from(signedTx);

    const parsedChainId = parsed.chainId == null ? null : BigInt(parsed.chainId);
    if (parsedChainId !== CHAIN_ID) throw new Error("Internal error: signed transaction has wrong chain id.");
    if ((parsed.nonce ?? -1) !== nonce) throw new Error("Internal error: nonce mismatch after signing.");
    if (!parsed.to) throw new Error("Internal error: missing 'to' after signing.");
    if (parsed.gasPrice == null) throw new Error("Internal error: missing legacy gasPrice after signing.");

    setText("txSummary", buildDecodedSummary(parsed));
    setText(
      "log",
      "Transaction signed successfully. Broadcast the signed hex using a trusted Ethereum mainnet RPC endpoint."
    );
  } catch (e) {
    const msg = e?.message ? e.message : String(e);
    setText("log", `Error: ${msg}`);
  } finally {
    btn.disabled = false;
  }
}

document.addEventListener("DOMContentLoaded", () => {
  const pkEl = $("fromPrivateKey");
  pkEl.addEventListener("input", () => {
    try {
      const w = new ethers.Wallet(safePrivateKey(pkEl.value));
      setValue("derivedFrom", w.address);
    } catch {
      setValue("derivedFrom", "");
    }
  });

  $("btnSign").addEventListener("click", createSignedTx);
  $("btnClear").addEventListener("click", clearAll);
  $("btnCopy").addEventListener("click", copySignedTx);

  clearAll();
});
